#!/bin/bash
# generate-speccpu2017.sh - Generate SPECcpu 2017 configuration file

# Exit codes
EXIT_SUCCESS=0
EXIT_FAILURE=-1

# Default
GCC_VERSION=
BITNESS=64
ITERATIONS=1
PARALLEL=1
SPECTYPE=specspeed
ARCH=`uname -m`

if [ "$SAMPLE_CYCLE_FACTOR" = "" ]; then
	export SAMPLE_CYCLE_FACTOR=36
fi
if [ "$SAMPLE_EVENT_FACTOR" = "" ]; then
	export SAMPLE_EVENT_FACTOR=4
fi

##
# usage - Print usage
usage() {
	echo "
generate-speccpu.sh (c) Mel Gorman 2008

Usage: generate-speccpu.sh [options]
  -h, --help   Print this usage message
  --gcc        GCC version (Default: default)
  --bitness    32/64 bitness (Default: $BITNESS)
  --iterations Iterations to run speccpu
  --parallel   Number of copies or threads to use
  --type       Pick specrate or specspeed
"
	exit $1
}

##
# warning - Print a warning
# die - Exit with error message
warning() {
	echo "WARNING: $@"
}

die() {
	echo "FATAL: $@"
	exit $EXIT_FAILURE
}

##
# emit_header - Emit header
emit_header() {
	echo "# Autogenerated by generate-speccpu2017.sh"
	echo
}

##
# detect_base - Detect base configuration
# emit_base - Emit the base SPEC configuration
detect_base() {
	ignore_errors=1
	tune=base
	output_format="txt,html,cfg,pdf,csv"
	ext=$ARCH-m$BITNESS-gcc`echo $GCC_VERSION | sed -e 's/\.//g'`
	reportable=1
	env_vars=1
	numcpus=$(grep -c '^processor' /proc/cpuinfo)
}

emit_base() {
	echo "## Base configuration"
	echo "%define label mmtests"
	echo "%define build_ncpus $numcpus"
	echo "%define os LINUX"
	echo "%define model -m$BITNESS"
	echo "%define bits $BITNESS"
	echo
	echo "command_add_redirect = 1"
	echo "flagsurl             = \$[top]/config/flags/gcc.xml"
	echo "ignore_errors        = $ignore_errors"
	echo "iterations           = $ITERATIONS"
	echo "label                = %{label}-m%{bits}"
	echo "line_width           = 1020"
	echo "log_line_width       = 1020"
	echo "makeflags            = --jobs=%{build_ncpus}"
	echo "mean_anyway          = 1"
	if [ $PARALLEL -gt 1 -a "$SPECTYPE" = "specspeed" ]; then
		echo "parallel_test        = 1"
	fi
	echo "output_format        = $output_format"
	echo "preenv               = 1"
	echo "reportable           = $reportable"
	echo "tune                 = $tune"
	echo
	if [ "$SPECTYPE" = "specspeed" ]; then
		echo "intspeed,fpspeed:"
		echo "    threads          = $PARALLEL"
	else
		echo "intrate,fprate:"
		echo "    copies           = $PARALLEL"
	fi
	echo
}

##
# emit_compiler - Print the compiler configuration
emit_compiler() {
	if [ "`which gcc$GCC_VERSION`" = "" ]; then die No gcc$GCC_VERSION; fi
	if [ "`which g++$GCC_VERSION`" = "" ]; then die No g++$GCC_VERSION; fi
	if [ "`which gfortran$GCC_VERSION`" = "" ]; then die No gfortran$GCC_VERSION; fi
	echo "## Compiler"
	echo "default:"
	echo "    CC                      = gcc$GCC_VERSION -std=c99 %{model}"
	echo "    CXX                     = g++$GCC_VERSION          %{model}"
	echo "    FC                      = gfortran$GCC_VERSION     %{model}"
	echo "    CC_VERSION_OPTION       = -v"
	echo "    CXX_VERSION_OPTION      = -v"
	echo "    FC_VERSION_OPTION       = -v"
	echo
	echo "default:"
	echo "    sw_base_ptrsize = $BITNESS-bit"
	echo "    sw_peak_ptrsize = $BITNESS-bit"
	echo
}

##
# cpuinfo_val - Output the given value of a cpuinfo field
cpuinfo_val() {
	grep "^$1" /proc/cpuinfo | awk -F": " '{print $2}' | head -1
}

##
# detect_mconf - Detect machine configuration
# emit_mconf - Emit machine HW configuration
# emit_onlymconf - Emit mconf and exit
detect_mconf() {
	# Common to all arches
	# Lookup primary cache information
	cache=/sys/devices/system/cpu/cpu0/cache
	pcache=
	for index in `ls /sys/devices/system/cpu/cpu0/cache`; do
		if [ "$pcache" != "" ]; then
			pcache="$pcache + "
		fi
		pcache="$pcache`cat $cache/$index/size 2>&1`"
		pcache="$pcache `cat $cache/$index/type 2>&1| head -c1`"
	done
	hw_memory=`free -m | grep ^Mem: | awk '{print $2}'`MB
	hw_cpus=`grep processor /proc/cpuinfo | wc -l`

	case "$ARCH" in
		i?86|x86_64|ia64)

			if [ "`which dmidecode`" = "" ]; then
				warning dmidecode is not in path, very limited info
			fi

			hw_manu=`dmidecode -s baseboard-manufacturer`
			hw_prod=`dmidecode -s baseboard-product-name`
			hw_vers=`dmidecode -s baseboard-version`

			hw_model="$hw_manu $hw_prod $hw_vers"
			hw_cpu_name=`cpuinfo_val "model name"`
			hw_cpu_mhz=`cpuinfo_val "cpu MHz"`
			hw_ncoresperchip=`cpuinfo_val "cpu cores"`
			hw_nchips=$(($hw_cpus/$hw_ncoresperchip))
			hw_ncores=$(($hw_cpus/$hw_nchips))
			hw_pcache=$pcache

			;;
		ppc64)
			hw_cpu_name=`cpuinfo_val cpu`
			hw_cpu_mhz=`cpuinfo_val "clock"`
			hw_pcache=$pcache
			;;
	esac
}

emit_mconf() {
	echo "## HW config"
	echo "intrate,intspeed,fprate,fpspeed:"
	echo "hw_vendor          = $hw_vendor"
	echo "tester             = mmtests"
	echo "test_sponsor       = mmtests"
	echo "license_num        = nnn"
	echo
	echo "intrate,intspeed,fprate,fpspeed:"
	echo "hw_avail           = Jun 2017"
	echo "sw_avail           = Jun 2017"
	echo
	echo "intrate,intspeed,fprate,fpspeed:"
	echo "hw_cpu_nominal_mhz = $hw_cpu_mhz"
	echo "hw_cpu_max_mhz     = $hw_cpu_mhz"
	echo "hw_model           = $hw_model"
	echo "hw_ncores          = $hw_ncores"
	echo "hw_nthreadspercore = $hw_nthreadspercore"
	echo "hw_ncpuorder       = "
	echo "hw_other           = "
	echo "hw_pcache          = $hw_pcache"
	echo "hw_scache          = $hw_scache"
	echo "hw_tcache          = $hw_tcache"
	echo "hw_ocache          = $hw_ocache"
	echo "fw_bios            ="
	echo "sw_other           ="
	echo

}

emit_sconf() {
	echo "## SW config"
	echo "default:"
	echo "    sw_compiler001     = C/C++/Fortran: Version $GCC_VERSION of GCC, the"
	echo "    sw_compiler002     = GNU Compiler Collection"
	echo
}

##
# detect_portability - detect portability flags
# emit_portall - Emit portability flags for all
detect_portability() {
	port_os=SPEC_CPU_LINUX

	case "$BITNESS" in
	64)
		port_all=SPEC_LP64
		;;
	32)
		port_all=_FILE_OFFSET_BITS=64
		;;
	esac
}

emit_portall() {
	echo "## Portability flags"
	echo "default:"
	echo "    EXTRA_PORTABILITY          = -D$port_all"

	VECTORISE=-fno-tree-loop-vectorize
	MAJOR=`gcc${GCC_VERSION} --version | head -1 | awk '{print $NF}' | awk -F . '{print $1}'`
	if [ $MAJOR -lt 6 ]; then
		VECTORISE=
	fi

	cat <<EOF
500.perlbench_r,600.perlbench_s:  #lang='C'
%if %{bits} == 32
%   define suffix IA32
%else
%   define suffix X64
%endif
   PORTABILITY    = -DSPEC_%{os}_%{suffix}

521.wrf_r,621.wrf_s:  #lang='F,C'
   CPORTABILITY  = -DSPEC_CASE_FLAG
   FPORTABILITY  = -fconvert=big-endian

523.xalancbmk_r,623.xalancbmk_s:  #lang='CXX'
   PORTABILITY   = -DSPEC_%{os}

526.blender_r:  #lang='CXX,C'
   PORTABILITY   = -funsigned-char -DSPEC_LINUX

527.cam4_r,627.cam4_s:  #lang='F,C'
   PORTABILITY   = -DSPEC_CASE_FLAG

628.pop2_s:  #lang='F,C'
   CPORTABILITY    = -DSPEC_CASE_FLAG
   FPORTABILITY    = -fconvert=big-endian

%if %{bits} == 32
   intspeed,fpspeed:
   #
   # Many of the speed benchmarks (6nn.benchmark_s) do not fit in 32 bits
   # If you wish to run SPECint2017_speed or SPECfp2017_speed, please use
   #
   #     runcpu --define bits=64
   #
   fail_build = 1
%else
   intspeed,fpspeed:
      EXTRA_OPTIMIZE = -fopenmp -DSPEC_OPENMP
   fpspeed:
      #
      # 627.cam4 needs a big stack; the preENV will apply it to all
      # benchmarks in the set, as required by the rules.
      #
      preENV_OMP_STACKSIZE = 120M
%endif

default=base:		# flags for all base
   OPTIMIZE		= -g -O3 -march=native -fno-unsafe-math-optimizations $VECTORISE
   EXTRA_CXXOPTIMIZE	= -std=c++03

intrate,intspeed=base: # flags for integer base
%if %{bits} == 32
    EXTRA_COPTIMIZE = -fgnu89-inline -fno-strict-aliasing
%else
    EXTRA_COPTIMIZE = -fno-strict-aliasing
    LDCFLAGS        = -z muldefs
%endif

default=peak:
   basepeak = yes
EOF
	echo
}

##
# emit_footer - Print the end
emit_footer() {
	echo "__MD5__"
}

# Parse the arguments
OPTARGS=`getopt -o h --long help,gcc,bitness:,iterations:,parallel:,spectype: -n generate-speccpu2017.sh -- "$@"`
eval set -- "$OPTARGS"
while [ "$1" != "" ] && [ "$1" != "--" ]; do
	case "$1" in
	-h|--help)
		usage $EXIT_SUCCESS;
		;;
	--gcc)
		GCC_VERSION=$2
		shift 2
		;;
	--bitness)
		BITNESS=$2
		shift 2
		;;
	--iterations)
		ITERATIONS=$2
		shift 2
		;;
	--parallel)
		PARALLEL=$2
		shift 2
		;;
	--spectype)
		if [ "$2" = "specspeed" -o "$2" = "specrate" ]; then
			SPECTYPE=$2
		else
			WHICH=`echo $2 | head -c 1`
			if [ "$WHICH" = "6" ]; then
				SPECTYPE=specspeed
			else
				SPECTYPE=specrate
			fi
		fi
		shift 2
	esac
done

# Automatic detection
detect_base
detect_mconf
detect_portability

# Generate a spec file
emit_header
emit_base
emit_compiler
emit_portall
emit_sconf
emit_mconf
emit_footer
