#!/usr/bin/perl
# compare-mmtests.pl - Compare results from an MM Tests directory
# 
# Copyright: SUSE Labs, 2012
# Author:    Mel Gorman, 2012

use FindBin qw($Bin);
use lib "$Bin/lib";

use Getopt::Long;
use Pod::Usage;
use VMR::Report;
use VMR::Stat;
use MMTests::Compare;
use MMTests::CompareFactory;
use MMTests::Extract;
use MMTests::ExtractFactory;
use MMTests::Monitor;
use MMTests::MonitorFactory;
use strict;

# Option variable
my ($opt_verbose);
my ($opt_help, $opt_manual);
my ($opt_reportDirectory);
my ($opt_printHeader, $opt_printExtra);
my ($opt_subheading, $opt_format);
my ($opt_names, $opt_benchmark);
my ($opt_monitor, $opt_hideCompare);
GetOptions(
	'verbose|v'		=> \$opt_verbose,
	'help|h'		=> \$opt_help,
	'--print-header'	=> \$opt_printHeader,
	'--print-extra'		=> \$opt_printExtra,
	'--print-monitor=s'	=> \$opt_monitor,
	'--no-compare'		=> \$opt_hideCompare,
	'--sub-heading=s'	=> \$opt_subheading,
	'--format=s'		=> \$opt_format,
	'n|names=s'		=> \$opt_names,
	'b|benchmark=s'		=> \$opt_benchmark,
	'manual'		=> \$opt_manual,
	'directory|d=s'		=> \$opt_reportDirectory,
);
setVerbose if $opt_verbose;
pod2usage(-exitstatus => 0, -verbose => 0) if $opt_help;
pod2usage(-exitstatus => 0, -verbose => 2) if $opt_manual;

# Sanity check directory
if (! -d $opt_reportDirectory) {
	printWarning("Report directory $opt_reportDirectory does not exist or was not specified.");
	pod2usage(-exitstatus => -1, -verbose => 0);
}

my @extractModules;
my $nrModules = 0;
if (!defined($opt_monitor)) {
	# Instantiate extract handlers for the requested type for the benchmark
	my $extractFactory = MMTests::ExtractFactory->new();
	for my $name (split /,/, $opt_names) {
		printVerbose("Loading extract $opt_benchmark $name\n");
		eval {
			my $reportDirectory = "$opt_reportDirectory/$opt_benchmark-$name";
			$extractModules[$nrModules] = $extractFactory->loadModule($opt_benchmark, $reportDirectory, $name);
			$extractModules[$nrModules]->extractReport($reportDirectory, $name);
			$extractModules[$nrModules++]->extractSummary($opt_subheading);
		} or do {
			printWarning("Failed to load module for benchmark $opt_benchmark: $name\n$@");
			$#extractModules -= 1;
		}
	};
} else {
	$opt_hideCompare = 1;
	my $extractFactory = MMTests::MonitorFactory->new();
	for my $name (split /,/, $opt_names) {
		printVerbose("Loading extract $opt_benchmark $name\n");
		eval {
			my $reportDirectory = $opt_reportDirectory;
			$extractModules[$nrModules] = $extractFactory->loadModule($opt_monitor, $reportDirectory, $name);
			$extractModules[$nrModules]->extractReport($reportDirectory, $name, $opt_benchmark, $opt_subheading, 1);
			$extractModules[$nrModules++]->extractSummary($opt_subheading);
		} or do {
			printWarning("Failed to load module for benchmark $opt_benchmark, $name\n$@");
			$#extractModules -= 1;
		}
	};
}
	
printVerbose("Loaded $nrModules extract modules\n");

# Instantiate comparison for the requested type for the benchmark
my $compareFactory = MMTests::CompareFactory->new();
my $compareModule;
printVerbose("Loading compare $opt_benchmark\n");
eval {
	if ($opt_monitor) {
		$compareModule = $compareFactory->loadModule("cputime", $opt_format, \@extractModules);
	} else {
		$compareModule = $compareFactory->loadModule($opt_benchmark, $opt_format, \@extractModules);
	}
} or do {
	printWarning("Failed to compare module for benchmark $opt_benchmark\n$@");
	exit(-1);
};
printVerbose("Loaded $nrModules compare modules\n");

$compareModule->extractComparison($opt_subheading, !$opt_hideCompare);
$compareModule->printComparison();

# Below this line is help and manual page information
__END__
=head1 NAME

compare-mmtests.pl - Compare results from an MM Tests result directory

=head1 SYNOPSIS

compare-mmtests.pl [options]

 Options:
 -d, --directory	Work log directory to extract data from
 -n, --names		Titles for the series if tests given to run-mmtests.sh
 -b, --benchmark	Benchmark to extract data for
 -v, --verbose		Verbose output
 --format		Output format
 --print-header		Print a header
 --print-extra		Print secondary data collected by the benchmark
 --sub-heading		Analyse just a sub-heading of the data, see manual page
 --manual		Print manual page
 --help			Print help message

=head1 OPTIONS

=over 8

=item B<-d, --directory>

Specifies the directory containing results generated by MM Tests.

=item B<n, --name>

The name of the test series as supplied to run-mmtests.sh. This might have
been a kernel version for example.

=item B<b, --benchmark>

The name of the benchmark to extract data from. For example, if a given
test ran kernbench and sysbench and the sysbench results were required
then specify "-b sysbench".

=item B<--format>

Output format for the report. Valid options are html and text. By default
the formatting is in plain text.

=item B<--print-header>

Print a header that briefly describes what each of the fields are.

=item B<--print-extra>

Print additional information collected by the benchmark. Some benchmarks
like dbench4 have a primary set of data such as throughput and latency
while running the benchmark. It also reports the average and max latency
of the commands sent to the server but this cannot be sanely represented
with the main data. Use --print-extra in cases like this to see.

=item B<--sub-heading>

For certain operation a sub-heading is required. For example, when extracting
CPUTime data and plotting it, it is necessary to specify if User, Sys,
Elapsed or CPU time is being plotted. In general --print-header can be
used to get a lot of the headings to pass to --sub-heading.

=item B<--no-compare>

It does not always make sense to use the default comparison operators. Use
this switch to hide them if they are deemed to be unnecessary. This can be
the case when comparing ratios for example or many of the monitors.

=item B<--help>

Print a help message and exit

=item B<-v, --verbose>

Be verbose in the output.

=back

=head1 DESCRIPTION

No detailed description available.

=head1 AUTHOD

Written by Mel Gorman <mgorman@suse.de>

=head1 REPORTING BUGS

Report bugs to the author.

=cut
