/*
 * Copyright (C) 2016 Freie Universität Berlin
 *
 * This file is subject to the terms and conditions of the GNU Lesser
 * General Public License v2.1. See the file LICENSE in the top level
 * directory for more details.
 */

/**
 * @ingroup     sys_mineplex
 * @{
 *
 * @file
 * @brief       This file contains the actual font data table
 *
 * @author      Hauke Petersen <hauke.petersen@fu-berlin.de>
 *
 * @}
 */

#include "mineplex.h"

/**
 * @brief   The first and the last ASCII character contained in the table
 */
#define ASCII_MIN               (0x20)
#define ASCII_MAX               (0x7e)

/**
 * @brief   Table containing the Mineplex encoded ASCII characters
 */
static const uint8_t mineplex[][MINEPLEX_CHAR_H] = {
    { 0x00, 0x00, 0x00, 0x00, 0x00 },   /* 20 SPACE*/
    { 0x02, 0x02, 0x02, 0x00, 0x02 },   /* 21 ! */
    { 0x0a, 0x0a, 0x00, 0x00, 0x00 },   /* 22 " */
    { 0x0a, 0x1f, 0x0a, 0x1f, 0x0a },   /* 23 # */
    { 0x1f, 0x05, 0x1f, 0x14, 0x1f },   /* 24 $ */
    { 0x11, 0x08, 0x04, 0x02, 0x11 },   /* 25 % */
    { 0x02, 0x05, 0x0e, 0x05, 0x0e },   /* 26 & */
    { 0x04, 0x04, 0x00, 0x00, 0x00 },   /* 27 ' */
    { 0x0c, 0x02, 0x02, 0x02, 0x0c },   /* 28 ( */
    { 0x06, 0x08, 0x08, 0x08, 0x06 },   /* 29 ) */
    { 0x00, 0x0a, 0x04, 0x0a, 0x00 },   /* 2a * */
    { 0x00, 0x04, 0x0e, 0x04, 0x00 },   /* 2b + */
    { 0x00, 0x00, 0x00, 0x02, 0x02 },   /* 2c , */
    { 0x00, 0x00, 0x0e, 0x00, 0x00 },   /* 2d - */
    { 0x00, 0x00, 0x00, 0x00, 0x02 },   /* 2e . */
    { 0x10, 0x08, 0x04, 0x02, 0x01 },   /* 2f / */
    { 0x06, 0x09, 0x09, 0x09, 0x06 },   /* 30 0 */
    { 0x06, 0x04, 0x04, 0x04, 0x0e },   /* 31 1 */
    { 0x06, 0x09, 0x04, 0x02, 0x0f },   /* 32 2 */
    { 0x06, 0x09, 0x04, 0x09, 0x06 },   /* 33 3 */
    { 0x09, 0x09, 0x0f, 0x08, 0x08 },   /* 34 4 */
    { 0x0f, 0x01, 0x07, 0x08, 0x07 },   /* 35 5 */
    { 0x0e, 0x01, 0x07, 0x09, 0x06 },   /* 36 6 */
    { 0x0f, 0x08, 0x08, 0x08, 0x08 },   /* 37 7 */
    { 0x06, 0x09, 0x06, 0x09, 0x06 },   /* 38 8 */
    { 0x06, 0x09, 0x0e, 0x08, 0x07 },   /* 39 9 */
    { 0x00, 0x02, 0x00, 0x00, 0x02 },   /* 3a : */
    { 0x00, 0x02, 0x00, 0x02, 0x02 },   /* 3b ; */
    { 0x08, 0x04, 0x02, 0x04, 0x08 },   /* 3c < */
    { 0x00, 0x0f, 0x00, 0x0f, 0x00 },   /* 3d = */
    { 0x02, 0x04, 0x08, 0x04, 0x02 },   /* 3e > */
    { 0x06, 0x09, 0x04, 0x00, 0x04 },   /* 3f ? */
    { 0x0e, 0x19, 0x1d, 0x15, 0x0e },   /* 40 @ */
    { 0x0f, 0x09, 0x0f, 0x09, 0x09 },   /* 41 A */
    { 0x07, 0x09, 0x07, 0x09, 0x07 },   /* 42 B */
    { 0x0f, 0x01, 0x01, 0x01, 0x0f },   /* 43 C */
    { 0x07, 0x09, 0x09, 0x09, 0x07 },   /* 44 D */
    { 0x0f, 0x01, 0x07, 0x01, 0x0f },   /* 45 E */
    { 0x0f, 0x01, 0x07, 0x01, 0x01 },   /* 46 F */
    { 0x0f, 0x01, 0x0d, 0x09, 0x0f },   /* 47 G */
    { 0x09, 0x09, 0x0f, 0x09, 0x09 },   /* 48 H */
    { 0x0e, 0x04, 0x04, 0x04, 0x0e },   /* 49 I */
    { 0x08, 0x08, 0x08, 0x09, 0x06 },   /* 4a J */
    { 0x09, 0x05, 0x03, 0x05, 0x09 },   /* 4b K */
    { 0x01, 0x01, 0x01, 0x01, 0x0f },   /* 4c L */
    { 0x1f, 0x15, 0x15, 0x11, 0x11 },   /* 4d M */
    { 0x09, 0x0b, 0x0d, 0x09, 0x09 },   /* 4e N */
    { 0x0f, 0x09, 0x09, 0x09, 0x0f },   /* 4f O */
    { 0x0f, 0x09, 0x0f, 0x01, 0x01 },   /* 50 P */
    { 0x0f, 0x09, 0x09, 0x05, 0x0b },   /* 51 Q */
    { 0x0f, 0x09, 0x07, 0x09, 0x09 },   /* 52 R */
    { 0x0f, 0x01, 0x0f, 0x08, 0x0f },   /* 53 S */
    { 0x1f, 0x04, 0x04, 0x04, 0x04 },   /* 54 T */
    { 0x09, 0x09, 0x09, 0x09, 0x0f },   /* 55 U */
    { 0x09, 0x09, 0x09, 0x09, 0x06 },   /* 56 V */
    { 0x11, 0x11, 0x15, 0x15, 0x1f },   /* 57 W */
    { 0x11, 0x0a, 0x04, 0x0a, 0x11 },   /* 58 X */
    { 0x11, 0x0a, 0x04, 0x04, 0x04 },   /* 59 Y */
    { 0x1f, 0x08, 0x04, 0x02, 0x1f },   /* 5a Z */
    { 0x0e, 0x02, 0x02, 0x02, 0x0e },   /* 5b [ */
    { 0x01, 0x02, 0x04, 0x08, 0x10 },   /* 5c \ */
    { 0x0e, 0x08, 0x08, 0x08, 0x0e },   /* 5d ] */
    { 0x04, 0x0a, 0x00, 0x00, 0x00 },   /* 5e ^ */
    { 0x00, 0x00, 0x00, 0x00, 0x0f },   /* 5f _ */
    { 0x06, 0x08, 0x00, 0x00, 0x00 },   /* 60 ` */
    { 0x0f, 0x09, 0x0f, 0x09, 0x09 },   /* 61 a */
    { 0x07, 0x09, 0x07, 0x09, 0x07 },   /* 62 b */
    { 0x0f, 0x01, 0x01, 0x01, 0x0f },   /* 63 c */
    { 0x07, 0x09, 0x09, 0x09, 0x07 },   /* 64 d */
    { 0x0f, 0x01, 0x07, 0x01, 0x0f },   /* 65 e */
    { 0x0f, 0x01, 0x07, 0x01, 0x01 },   /* 66 f */
    { 0x0f, 0x01, 0x0d, 0x09, 0x0f },   /* 67 g */
    { 0x09, 0x09, 0x0f, 0x09, 0x09 },   /* 68 h */
    { 0x0e, 0x04, 0x04, 0x04, 0x0e },   /* 69 i */
    { 0x08, 0x08, 0x08, 0x09, 0x06 },   /* 6a j */
    { 0x09, 0x05, 0x03, 0x05, 0x09 },   /* 6b k */
    { 0x01, 0x01, 0x01, 0x01, 0x0f },   /* 6c l */
    { 0x1f, 0x15, 0x15, 0x11, 0x11 },   /* 6d m */
    { 0x09, 0x0b, 0x0d, 0x09, 0x09 },   /* 6e n */
    { 0x0f, 0x09, 0x09, 0x09, 0x0f },   /* 6f o */
    { 0x0f, 0x09, 0x0f, 0x01, 0x01 },   /* 70 p */
    { 0x0f, 0x09, 0x09, 0x05, 0x0b },   /* 71 q */
    { 0x0f, 0x09, 0x07, 0x09, 0x09 },   /* 72 r */
    { 0x0f, 0x01, 0x0f, 0x08, 0x0f },   /* 73 s */
    { 0x1f, 0x04, 0x04, 0x04, 0x04 },   /* 74 t */
    { 0x09, 0x09, 0x09, 0x09, 0x0f },   /* 75 u */
    { 0x09, 0x09, 0x09, 0x09, 0x06 },   /* 76 v */
    { 0x11, 0x11, 0x15, 0x15, 0x1f },   /* 77 w */
    { 0x11, 0x0a, 0x04, 0x0a, 0x11 },   /* 78 x */
    { 0x11, 0x0a, 0x04, 0x04, 0x04 },   /* 79 y */
    { 0x1f, 0x08, 0x04, 0x02, 0x1f },   /* 7a z */
    { 0x0c, 0x02, 0x03, 0x02, 0x0c },   /* 7b { */
    { 0x04, 0x04, 0x04, 0x04, 0x04 },   /* 7c | */
    { 0x03, 0x04, 0x0c, 0x04, 0x03 },   /* 7d } */
    { 0x00, 0x0a, 0x05, 0x00, 0x00 }    /* 7e ~ */
};

const uint8_t *mineplex_char(char c)
{
    if ((c < ASCII_MIN) || (c > ASCII_MAX)) {
        return mineplex[0];
    }
    return mineplex[((int)c) - ASCII_MIN];
}
