/*
 * Copyright (C) 2015 Eistec AB
 *
 * This file is subject to the terms and conditions of the GNU Lesser
 * General Public License v2.1. See the file LICENSE in the top level
 * directory for more details.
 */

/**
 * @ingroup     examples
 * @{
 *
 * @file
 * @brief       ARM DSP library test application
 *
 * @author      Joakim Nohlgård <joakim.nohlgard@eistec.se>
 *
 * @}
 */

#include <stdio.h>
#include <stdint.h>
#include <math.h>
#include "arm_math.h"

#define MAX_BLOCKSIZE   32
#define DELTA           (0.0000001f)
/* ----------------------------------------------------------------------
* Declare I/O buffers
* ------------------------------------------------------------------- */
float32_t wire1[MAX_BLOCKSIZE];
float32_t wire2[MAX_BLOCKSIZE];
float32_t wire3[MAX_BLOCKSIZE];
/* ----------------------------------------------------------------------
* Test input data for Floating point Variance example for 32-blockSize
* Generated by the MATLAB randn() function
* ------------------------------------------------------------------- */
float32_t testInput_f32[32] =
{
  -0.432564811528221,  -1.665584378238097,   0.125332306474831,   0.287676420358549,
  -1.146471350681464,   1.190915465642999,   1.189164201652103,  -0.037633276593318,
   0.327292361408654,   0.174639142820925,  -0.186708577681439,   0.725790548293303,
  -0.588316543014189,   2.183185818197101,  -0.136395883086596,   0.113931313520810,
   1.066768211359189,   0.059281460523605,  -0.095648405483669,  -0.832349463650022,
   0.294410816392640,  -1.336181857937804,   0.714324551818952,   1.623562064446271,
  -0.691775701702287,   0.857996672828263,   1.254001421602532,  -1.593729576447477,
  -1.440964431901020,   0.571147623658178,  -0.399885577715363,   0.689997375464345
};
/* ----------------------------------------------------------------------
* Declare Global variables
* ------------------------------------------------------------------- */
uint32_t blockSize = 32;
float32_t  refVarianceOut = 0.903941793931839;
/* ----------------------------------------------------------------------
* Variance calculation test
* ------------------------------------------------------------------- */
int main(void)
{
    arm_status status;
    float32_t mean;
    float32_t oneByBlockSize = 1.0 / (blockSize);
    float32_t variance;
    float32_t diff;
    status = ARM_MATH_SUCCESS;
    puts("ARM DSP lib test");
    puts("Note: This test is using 32 bit IEEE 754 floating point numbers,"
         "(24 bit mantissa, 7 bit exponent)");
    puts("Expect roughly 7 decimals precision on the result.");
    /* Calculation of mean value of input */
    /* x' = 1/blockSize * (x(0)* 1 + x(1) * 1 + ... + x(n-1) * 1) */
    /* Fill wire1 buffer with 1.0 value */
    arm_fill_f32(1.0, wire1, blockSize);
    /* Calculate the dot product of wire1 and wire2 */
    /* (x(0)* 1 + x(1) * 1 + ...+ x(n-1) * 1) */
    arm_dot_prod_f32(testInput_f32, wire1, blockSize, &mean);
    /* 1/blockSize * (x(0)* 1 + x(1) * 1 + ... + x(n-1) * 1)  */
    arm_mult_f32(&mean, &oneByBlockSize, &mean, 1);
    /* Calculation of variance value of input */
    /* (1/blockSize) * (x(0) - x') * (x(0) - x') + (x(1) - x') * (x(1) - x') + ... + (x(n-1) - x') * (x(n-1) - x') */
    /* Fill wire2 with mean value x' */
    arm_fill_f32(mean, wire2, blockSize);
    /* wire3 contains (x-x') */
    arm_sub_f32(testInput_f32, wire2, wire3, blockSize);
    /* wire2 contains (x-x') */
    arm_copy_f32(wire3, wire2, blockSize);
    /* (x(0) - x') * (x(0) - x') + (x(1) - x') * (x(1) - x') + ... + (x(n-1) - x') * (x(n-1) - x') */
    arm_dot_prod_f32(wire2, wire3, blockSize, &variance);
    /* Calculation of 1/blockSize */
    oneByBlockSize = 1.0 / (blockSize - 1);
    /* Calculation of variance */
    arm_mult_f32(&variance, &oneByBlockSize, &variance, 1);
    /* absolute value of difference between ref and test */
    diff = variance - refVarianceOut;
    /* Split into fractional and integral parts, since printing floats may not be supported on all platforms */
    float int_part;
    float frac_part = fabsf(modff(variance, &int_part));
    printf("      dsp: %3d.%09d\n", (int) int_part, (int) (frac_part * 1.0e9f + 0.5f));
    puts(  "reference:   0.903941793931839");
    frac_part = fabsf(modff(diff, &int_part));
    printf("     diff: %3d.%09d\n", (int) int_part, (int) (frac_part * 1.0e9f + 0.5f));
    /* Comparison of variance value with reference */
    if(fabsf(diff) > DELTA) {
        status = ARM_MATH_TEST_FAILURE;
    }
    if(status != ARM_MATH_SUCCESS) {
        puts("Test failed");
        while(1)
            ;
    }
    puts("Test done");
    while(1)
        ; /* main function does not return */
}
