package state

import (
	"time"
)

// PrunerObserver allows an external observer to be notified of certain events
// generated by the [Pruner].
type PrunerObserver interface {
	// PrunerStarted is called when the pruner's background pruning routine has
	// been started.
	PrunerStarted(interval time.Duration)
	// PrunerPrunedABCIRes is called after each successful pruning of ABCI results.
	PrunerPrunedABCIRes(prunedInfo *ABCIResponsesPrunedInfo)
	// PrunerPrunedBlocks is called after each successful pruning of blocks.
	PrunerPrunedBlocks(prunedInfo *BlocksPrunedInfo)
}

// BlocksPrunedInfo provides information about blocks pruned during a single
// run of the pruner.
type BlocksPrunedInfo struct {
	FromHeight int64 // The height from which blocks were pruned (inclusive).
	ToHeight   int64 // The height to which blocks were pruned (inclusive).
}

// ABCIResponsesPrunedInfo provides information about ABCI responses pruned
// during a single run of the pruner.
type ABCIResponsesPrunedInfo struct {
	FromHeight int64 // The height from which ABCI responses were pruned (inclusive).
	ToHeight   int64 // The height to which ABCI responses were pruned (inclusive).
}

// NoopPrunerObserver does nothing.
type NoopPrunerObserver struct{}

var _ PrunerObserver = NoopPrunerObserver{}

// PrunerPrunedABCIRes implements PrunerObserver.
func (NoopPrunerObserver) PrunerPrunedABCIRes(*ABCIResponsesPrunedInfo) {}

// PrunerPrunedBlocks implements PrunerObserver.
func (NoopPrunerObserver) PrunerPrunedBlocks(*BlocksPrunedInfo) {}

// PrunerStarted implements PrunerObserver.
func (NoopPrunerObserver) PrunerStarted(time.Duration) {}
