// Copyright 2021 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package com.google.devtools.build.lib.bazel.bzlmod;

import com.google.auto.value.AutoValue;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.devtools.build.lib.analysis.BlazeDirectories;
import com.google.devtools.build.lib.cmdline.LabelConstants;
import com.google.devtools.build.lib.events.ExtendedEventHandler;
import com.google.devtools.build.lib.packages.NoSuchPackageException;
import com.google.devtools.build.lib.packages.Package;
import com.google.devtools.build.lib.packages.Package.NameConflictException;
import com.google.devtools.build.lib.packages.PackageFactory;
import com.google.devtools.build.lib.packages.RuleFactory.InvalidRuleException;
import com.google.devtools.build.lib.packages.StarlarkNativeModule.ExistingRulesShouldBeNoOp;
import com.google.devtools.build.lib.vfs.Root;
import com.google.devtools.build.lib.vfs.RootedPath;
import java.util.HashMap;
import java.util.Map;
import net.starlark.java.eval.EvalException;
import net.starlark.java.eval.Starlark;
import net.starlark.java.eval.StarlarkSemantics;
import net.starlark.java.eval.StarlarkThread;
import net.starlark.java.syntax.Location;

/**
 * A context object that should be stored in a {@link StarlarkThread} for use during module
 * extension evaluation.
 */
public final class ModuleExtensionEvalStarlarkThreadContext {
  public void storeInThread(StarlarkThread thread) {
    thread.setThreadLocal(ModuleExtensionEvalStarlarkThreadContext.class, this);
    // The following is just a hack; see documentation there for an explanation.
    thread.setThreadLocal(ExistingRulesShouldBeNoOp.class, new ExistingRulesShouldBeNoOp());
  }

  public static ModuleExtensionEvalStarlarkThreadContext from(StarlarkThread thread) {
    return thread.getThreadLocal(ModuleExtensionEvalStarlarkThreadContext.class);
  }

  @AutoValue
  abstract static class PackageAndLocation {
    abstract Package getPackage();

    abstract Location getLocation();

    static PackageAndLocation create(Package pkg, Location location) {
      return new AutoValue_ModuleExtensionEvalStarlarkThreadContext_PackageAndLocation(
          pkg, location);
    }
  }

  private final String repoPrefix;
  private final PackageFactory packageFactory;
  private final BlazeDirectories directories;
  private final ExtendedEventHandler eventHandler;
  private final Map<String, PackageAndLocation> generatedRepos = new HashMap<>();

  public ModuleExtensionEvalStarlarkThreadContext(
      String repoPrefix,
      PackageFactory packageFactory,
      BlazeDirectories directories,
      ExtendedEventHandler eventHandler) {
    this.repoPrefix = repoPrefix;
    this.packageFactory = packageFactory;
    this.directories = directories;
    this.eventHandler = eventHandler;
  }

  /** The string that should be prefixed to the names of all repos generated by this extension. */
  public String getRepoPrefix() {
    return repoPrefix;
  }

  public void createRepo(
      String internalRepoName,
      BzlmodRepoRuleCreator repoRuleCreator,
      Map<String, Object> kwargs,
      StarlarkSemantics starlarkSemantics,
      Location location)
      throws InterruptedException, EvalException {
    PackageAndLocation conflict = generatedRepos.get(internalRepoName);
    if (conflict != null) {
      throw Starlark.errorf(
          "A repo named %s is already generated by this module extension at %s",
          internalRepoName, conflict.getLocation());
    }

    Package.Builder packageBuilder =
        packageFactory.newExternalPackageBuilder(
            RootedPath.toRootedPath(
                Root.fromPath(directories.getWorkspace()),
                LabelConstants.MODULE_DOT_BAZEL_FILE_NAME),
            "dummy_name",
            starlarkSemantics);
    try {
      repoRuleCreator.createAndAddRule(packageBuilder, starlarkSemantics, kwargs, eventHandler);
    } catch (InvalidRuleException | NameConflictException e) {
      throw Starlark.errorf("%s", e.getMessage());
    }
    try {
      generatedRepos.put(
          internalRepoName, PackageAndLocation.create(packageBuilder.build(), location));
    } catch (NoSuchPackageException e) {
      // This will never happen!
      throw new IllegalStateException(e);
    }
  }

  /**
   * Returns the repos generated by the extension so far. The key is the "internal name" (as
   * specified by the extension) of the repo, and the value is the package containing (only) the
   * repo rule.
   */
  public ImmutableMap<String, Package> getGeneratedRepos() {
    return ImmutableMap.copyOf(
        Maps.transformValues(generatedRepos, PackageAndLocation::getPackage));
  }
}
