// Copyright (C) 2019-2025 Algorand, Inc.
// This file is part of go-algorand
//
// go-algorand is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as
// published by the Free Software Foundation, either version 3 of the
// License, or (at your option) any later version.
//
// go-algorand is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with go-algorand.  If not, see <https://www.gnu.org/licenses/>.

package execpool

import (
	"context"
	"sync"

	"github.com/algorand/go-algorand/util"
)

// A backlog for an execution pool. The typical usage of this is to
// create non-blocking queue which would get executed once the execution pool is ready to accept new
// tasks.
type backlog struct {
	pool      ExecutionPool
	wg        sync.WaitGroup
	buffer    chan backlogItemTask
	ctx       context.Context
	ctxCancel context.CancelFunc
	owner     interface{}
	priority  Priority
}

type backlogItemTask struct {
	enqueuedTask
	priority Priority
}

// BacklogPool supports all the ExecutionPool functions plus few more that tests the pending tasks.
type BacklogPool interface {
	ExecutionPool
	EnqueueBacklog(enqueueCtx context.Context, t ExecFunc, arg interface{}, out chan interface{}) error
	BufferSize() (length, capacity int)
}

// MakeBacklog creates a backlog
func MakeBacklog(execPool ExecutionPool, backlogSize int, priority Priority, owner interface{}, profLabels ...string) BacklogPool {
	if backlogSize < 0 {
		return nil
	}
	bl := &backlog{
		pool:     execPool,
		owner:    owner,
		priority: priority,
	}
	bl.ctx, bl.ctxCancel = context.WithCancel(context.Background())
	if bl.pool == nil {
		// create one internally.
		bl.pool = MakePool(bl, append(profLabels, "execpool", "internal")...)
	}
	if backlogSize == 0 {
		// use the number of cpus in the system.
		backlogSize = bl.pool.GetParallelism()
	}
	bl.buffer = make(chan backlogItemTask, backlogSize)

	bl.wg.Add(1)
	go bl.worker(profLabels)
	return bl
}

func (b *backlog) GetParallelism() int {
	return b.pool.GetParallelism()
}

// Enqueue enqueues a single task into the backlog
func (b *backlog) Enqueue(enqueueCtx context.Context, t ExecFunc, arg interface{}, priority Priority, out chan interface{}) error {
	select {
	case b.buffer <- backlogItemTask{
		enqueuedTask: enqueuedTask{
			execFunc: t,
			arg:      arg,
			out:      out,
		},
		priority: priority,
	}:
		return nil
	case <-enqueueCtx.Done():
		return enqueueCtx.Err()
	case <-b.ctx.Done():
		return b.ctx.Err()
	}
}

// BufferSize returns the length and the capacity of the buffer
func (b *backlog) BufferSize() (length, capacity int) {
	return len(b.buffer), cap(b.buffer)
}

// EnqueueBacklog enqueues a single task into the backlog
func (b *backlog) EnqueueBacklog(enqueueCtx context.Context, t ExecFunc, arg interface{}, out chan interface{}) error {
	select {
	case b.buffer <- backlogItemTask{
		enqueuedTask: enqueuedTask{
			execFunc: t,
			arg:      arg,
			out:      out,
		},
		priority: b.priority,
	}:
		return nil
	case <-enqueueCtx.Done():
		return enqueueCtx.Err()
	case <-b.ctx.Done():
		return b.ctx.Err()
	}
}

// Shutdown shuts down the backlog.
func (b *backlog) Shutdown() {
	b.ctxCancel()
	// NOTE: Do not close(b.buffer) because there's no good way to ensure Enqueue*() won't write to it and panic. Just let it be garbage collected.
	b.wg.Wait()
	if b.pool.GetOwner() == b {
		b.pool.Shutdown()
	}
}

func (b *backlog) worker(profLabels []string) {
	var t backlogItemTask
	var ok bool
	defer b.wg.Done()
	util.SetGoroutineLabels(profLabels...)

	for {

		select {
		case t, ok = <-b.buffer:
		case <-b.ctx.Done():
			return
		}

		if !ok {
			return
		}

		if b.pool.Enqueue(b.ctx, t.execFunc, t.arg, t.priority, t.out) != nil {
			break
		}
	}
}

func (b *backlog) GetOwner() interface{} {
	return b.owner
}
