/*
 * FilterPreferencePanel.java
 *
 * Created on January 10, 2007, 6:31 PM
 */
package org.jmeld.ui.settings;

import org.jmeld.settings.*;
import org.jmeld.settings.util.*;
import org.jmeld.ui.swing.table.*;
import org.jmeld.ui.swing.table.util.*;
import org.jmeld.ui.util.*;
import org.jmeld.util.conf.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.List;

/**
 *
 * @author  kees
 */
public class FilterSettingsPanel
       extends FilterSettingsForm
       implements ConfigurationListenerIF
{
  JMTableModel filterTableModel;
  JMTableModel filterRuleTableModel;

  public FilterSettingsPanel()
  {
    init();
    initConfiguration();

    JMeldSettings.getInstance().addConfigurationListener(this);
  }

  private void init()
  {
    ListSelectionModel selectionModel;

    filterTableModel = getFilterTableModel();
    filterTable.setModel(filterTableModel);

    filterTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    selectionModel = filterTable.getSelectionModel();
    selectionModel.addListSelectionListener(getFilterSelectionAction());

    filterRuleTableModel = getFilterRuleTableModel(0);
    filterRuleTable.setModel(filterRuleTableModel);
    filterRuleTable.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);

    newFilterButton.addActionListener(getNewFilterAction());
    deleteFilterButton.addActionListener(getDeleteFilterAction());
    newFilterRuleButton.addActionListener(getNewFilterRuleAction());
    deleteFilterRuleButton.addActionListener(getDeleteFilterRuleAction());
  }

  private JMTableModel getFilterTableModel()
  {
    return new FilterTableModel();
  }

  private JMTableModel getFilterRuleTableModel(int filterIndex)
  {
    return new FilterRuleTableModel(filterIndex);
  }

  private ListSelectionListener getFilterSelectionAction()
  {
    return new ListSelectionListener()
      {
        public void valueChanged(ListSelectionEvent e)
        {
          int    rowIndex;
          Object value;

          if (e.getValueIsAdjusting())
          {
            return;
          }

          rowIndex = filterTable.getSelectedRow();
          value = filterTableModel.getValueAt(rowIndex, 0);

          filterNameLabel.setText(value.toString());
          filterRuleTableModel = getFilterRuleTableModel(rowIndex);
          filterRuleTable.setModel(filterRuleTableModel);
          filterRuleTable.doLayout();
        }
      };
  }

  private ActionListener getNewFilterAction()
  {
    return new ActionListener()
      {
        public void actionPerformed(ActionEvent ae)
        {
          getFilterSettings().addFilter(new Filter("Untitled"));
          filterTableModel.fireTableDataChanged();
        }
      };
  }

  private ActionListener getDeleteFilterAction()
  {
    return new ActionListener()
      {
        public void actionPerformed(ActionEvent ae)
        {
          getFilterSettings().removeFilter(getSelectedFilter());
          filterTableModel.fireTableDataChanged();
        }
      };
  }

  private ActionListener getNewFilterRuleAction()
  {
    return new ActionListener()
      {
        public void actionPerformed(ActionEvent ae)
        {
          Filter     filter;
          FilterRule newRule;
          FilterRule selectedFilterRule;

          filter = getSelectedFilter();
          if (filter == null)
          {
            return;
          }

          newRule = new FilterRule("Untitled", FilterRule.Rule.excludes, "",
              true);

          selectedFilterRule = getSelectedFilterRule();
          if (selectedFilterRule != null)
          {
            newRule.setDescription(selectedFilterRule.getDescription());
            newRule.setRule(selectedFilterRule.getRule());
            filter.insertRule(selectedFilterRule, newRule);
          }
          else
          {
            filter.addRule(newRule);
          }

          filterRuleTableModel.fireTableDataChanged();
        }
      };
  }

  private ActionListener getDeleteFilterRuleAction()
  {
    return new ActionListener()
      {
        public void actionPerformed(ActionEvent ae)
        {
          Filter     filter;
          FilterRule rule;

          filter = getSelectedFilter();
          if (filter == null)
          {
            return;
          }

          rule = getSelectedFilterRule();
          if (rule == null)
          {
            return;
          }

          filter.removeRule(rule);
          filterRuleTableModel.fireTableDataChanged();
        }
      };
  }

  public void configurationChanged()
  {
    initConfiguration();
  }

  private void initConfiguration()
  {
  }

  private class FilterTableModel
         extends JMTableModel
  {
    Column nameColumn;

    FilterTableModel()
    {
      nameColumn = addColumn("name", null, "Name", String.class, -1, true, null);
    }

    public int getRowCount()
    {
      return getFilterSettings().getFilters().size();
    }

    public void setValueAt(
      Object value,
      int    rowIndex,
      Column column)
    {
      Filter filter;

      filter = getFilter(rowIndex);
      if (filter != null)
      {
        if (column == nameColumn)
        {
          filter.setName((String) value);
        }
      }
    }

    public Object getValueAt(
      int    rowIndex,
      Column column)
    {
      Filter filter;

      filter = getFilter(rowIndex);
      if (filter != null)
      {
        if (column == nameColumn)
        {
          return filter.getName();
        }
      }

      return "";
    }

    private Filter getFilter(int rowIndex)
    {
      return getFilters().get(rowIndex);
    }
  }

  private class FilterRuleTableModel
         extends JMTableModel
  {
    private int    filterIndex;
    private Column activeColumn;
    private Column descriptionColumn;
    private Column ruleColumn;
    private Column patternColumn;

    public FilterRuleTableModel(int filterIndex)
    {
      this.filterIndex = filterIndex;

      activeColumn = addColumn("active", null, "Active", Boolean.class, 5,
          true, null);
      descriptionColumn = addColumn("description", null, "Description",
          String.class, 15, true, null);
      ruleColumn = addColumn("rule", null, "Rule", FilterRule.Rule.class, 10,
          true, null);
      ruleColumn.setRenderer(new JMComboBoxRenderer(FilterRule.Rule.values()));
      ruleColumn.setEditor(new JMComboBoxEditor(FilterRule.Rule.values()));
      patternColumn = addColumn("pattern", null, "Pattern", String.class, -1,
          true, null);
    }

    public int getRowCount()
    {
      return getRules(filterIndex).size();
    }

    public void setValueAt(
      Object value,
      int    rowIndex,
      Column column)
    {
      FilterRule rule;

      rule = getRule(rowIndex);
      if (rule != null)
      {
        if (column == activeColumn)
        {
          rule.setActive((Boolean) value);
        }

        if (column == descriptionColumn)
        {
          rule.setDescription((String) value);
        }

        if (column == ruleColumn)
        {
          rule.setRule((FilterRule.Rule) value);
        }

        if (column == patternColumn)
        {
          rule.setPattern((String) value);
        }
      }
    }

    public Object getValueAt(
      int    rowIndex,
      Column column)
    {
      FilterRule rule;

      rule = getRule(rowIndex);
      if (rule != null)
      {
        if (column == activeColumn)
        {
          return rule.getActive();
        }

        if (column == descriptionColumn)
        {
          return rule.getDescription();
        }

        if (column == ruleColumn)
        {
          return rule.getRule();
        }

        if (column == patternColumn)
        {
          return rule.getPattern();
        }
      }

      return "??";
    }

    private FilterRule getRule(int rowIndex)
    {
      return getRules(filterIndex).get(rowIndex);
    }
  }

  private Filter getSelectedFilter()
  {
    int rowIndex;

    rowIndex = filterTable.getSelectedRow();
    if (rowIndex < 0 || rowIndex > getFilters().size())
    {
      return null;
    }

    return getFilters().get(rowIndex);
  }

  private FilterRule getSelectedFilterRule()
  {
    Filter filter;
    int    rowIndex;

    filter = getSelectedFilter();
    if (filter == null)
    {
      return null;
    }

    rowIndex = filterRuleTable.getSelectedRow();
    if (rowIndex < 0 || rowIndex > filter.getRules().size())
    {
      return null;
    }

    return filter.getRules().get(rowIndex);
  }

  private List<FilterRule> getRules(int filterIndex)
  {
    int size;

    size = getFilters().size();
    if (filterIndex < 0 || filterIndex >= size)
    {
      return Collections.emptyList();
    }

    return getFilters().get(filterIndex).getRules();
  }

  private List<Filter> getFilters()
  {
    return getFilterSettings().getFilters();
  }

  private FilterSettings getFilterSettings()
  {
    return JMeldSettings.getInstance().getFilter();
  }
}
