"""Unit tests for all time series regressors."""

__maintainer__ = []


import numpy as np
from sklearn.utils._testing import set_random_state

from aeon.datasets import load_cardano_sentiment, load_covid_3month
from aeon.testing.expected_results.expected_regressor_outputs import (
    cardano_sentiment_preds,
    covid_3month_preds,
)
from aeon.testing.test_all_estimators import BaseFixtureGenerator, QuickTester
from aeon.testing.utils.estimator_checks import _assert_array_almost_equal


class RegressorFixtureGenerator(BaseFixtureGenerator):
    """Fixture generator for regression tests.

    Fixtures parameterized
    ----------------------
    estimator_class: estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
    estimator_instance: instance of estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
        instances are generated by create_test_instance class method
    scenario: instance of TestScenario
        ranges over all scenarios returned by retrieve_scenarios
    """

    # note: this should be separate from TestAllRegressors
    #   additional fixtures, parameters, etc should be added here
    #   TestAllRegressors should contain the tests only

    estimator_type_filter = "regressor"


class TestAllRegressors(RegressorFixtureGenerator, QuickTester):
    """Module level tests for all aeon regressors."""

    def test_regressor_against_expected_results(self, estimator_class):
        """Test classifier against stored results."""
        # we only use the first estimator instance for testing
        classname = estimator_class.__name__

        for data_name, data_dict, data_loader, data_seed in [
            ["Covid3Month", covid_3month_preds, load_covid_3month, 0],
            ["CardanoSentiment", cardano_sentiment_preds, load_cardano_sentiment, 0],
        ]:
            # retrieve expected predict output, and skip test if not available
            if classname in data_dict.keys():
                expected_preds = data_dict[classname]
            else:
                # skip test if no expected preds are registered
                continue

            # we only use the first estimator instance for testing
            estimator_instance = estimator_class.create_test_instance(
                parameter_set="results_comparison"
            )
            # set random seed if possible
            set_random_state(estimator_instance, 0)

            # load test data
            X_train, y_train = data_loader(split="train")
            X_test, y_test = data_loader(split="test")
            indices_train = np.random.RandomState(data_seed).choice(
                len(y_train), 10, replace=False
            )
            indices_test = np.random.RandomState(data_seed).choice(
                len(y_test), 10, replace=False
            )

            # train regressor and predict
            estimator_instance.fit(X_train[indices_train], y_train[indices_train])
            y_pred = estimator_instance.predict(X_test[indices_test])

            # assert predictions are the same
            _assert_array_almost_equal(
                y_pred,
                expected_preds,
                decimal=2,
                err_msg=f"Failed to reproduce results for {classname} on {data_name}",
            )

    def test_regressor_tags_consistent(self, estimator_class):
        """Test the tag X_inner_type is consistent with capability:unequal_length."""
        valid_types = {"np-list", "df-list", "pd-multivariate", "nested_univ"}
        unequal = estimator_class.get_class_tag("capability:unequal_length")
        if unequal:  # one of X_inner_types must be capable of storing unequal length
            internal_types = estimator_class.get_class_tag("X_inner_type")
            if isinstance(internal_types, str):
                assert internal_types in valid_types
            else:  # must be a list
                assert bool(set(internal_types) & valid_types)
        # Test can actually fit/predict with multivariate if tag is set
        multivariate = estimator_class.get_class_tag("capability:multivariate")
        if multivariate:
            X = np.random.random((10, 2, 20))
            y = np.random.random(10)
            inst = estimator_class.create_test_instance(parameter_set="default")
            inst.fit(X, y)
            inst.predict(X)

    def test_does_not_override_final_methods(self, estimator_class):
        """Test does not override final methods."""
        if "fit" in estimator_class.__dict__:
            raise ValueError(f"Classifier {estimator_class} overrides the method fit")
        if "predict" in estimator_class.__dict__:
            raise ValueError(
                f"Classifier {estimator_class} overrides the method " f"predict"
            )
