"""Unit tests for early classifier input output."""

__maintainer__ = []

import numpy as np
from sklearn.utils._testing import set_random_state

from aeon.datasets import load_basic_motions, load_unit_test
from aeon.testing.expected_results.expected_classifier_outputs import (
    basic_motions_proba,
    unit_test_proba,
)
from aeon.testing.test_all_estimators import BaseFixtureGenerator, QuickTester
from aeon.testing.utils.estimator_checks import _assert_array_almost_equal


class EarlyClassifierFixtureGenerator(BaseFixtureGenerator):
    """Fixture generator for early classifier tests.

    Fixtures parameterized
    ----------------------
    estimator_class: estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
    estimator_instance: instance of estimator inheriting from BaseObject
        ranges over estimator classes not excluded by EXCLUDE_ESTIMATORS, EXCLUDED_TESTS
        instances are generated by create_test_instance class method
    scenario: instance of TestScenario
        ranges over all scenarios returned by retrieve_scenarios
    """

    # note: this should be separate from TestAllEarlyClassifiers
    #   additional fixtures, parameters, etc should be added here
    #   TestAllEarlyClassifiers should contain the tests only

    estimator_type_filter = "early_classifier"


class TestAllEarlyClassifiers(EarlyClassifierFixtureGenerator, QuickTester):
    """Module level tests for all aeon classifiers."""

    def test_classifier_output(self, estimator_instance, scenario):
        """Test classifier outputs the correct data types and values.

        Test predict produces a np.array or pd.Series with only values seen in the train
        data, and that predict_proba probability estimates add up to one.
        """
        n_classes = scenario.get_tag("n_classes")
        X_new = scenario.args["predict"]["X"]
        y_train = scenario.args["fit"]["y"]
        y_pred, decisions = scenario.run(
            estimator_instance, method_sequence=["fit", "predict"]
        )

        # check predict
        assert isinstance(y_pred, np.ndarray)
        assert y_pred.shape == (X_new.shape[0],)
        assert np.all(np.isin(np.unique(y_pred), np.unique(y_train)))
        assert isinstance(decisions, np.ndarray)
        assert decisions.shape == (X_new.shape[0],)
        assert decisions.dtype == bool

        # predict and update methods should update the state info as an array
        assert isinstance(estimator_instance.get_state_info(), np.ndarray)

        # check predict proba (all classifiers have predict_proba by default)
        y_proba, decisions = scenario.run(
            estimator_instance, method_sequence=["predict_proba"]
        )
        assert isinstance(y_proba, np.ndarray)
        assert y_proba.shape == (X_new.shape[0], n_classes)
        np.testing.assert_allclose(y_proba.sum(axis=1), 1)
        assert isinstance(decisions, np.ndarray)
        assert decisions.shape == (X_new.shape[0],)
        assert decisions.dtype == bool

    def test_early_classifier_against_expected_results(self, estimator_class):
        """Test early classifier against stored results."""
        # we only use the first estimator instance for testing
        classname = estimator_class.__name__

        for data_name, data_dict, data_loader, data_seed in [
            ["UnitTest", unit_test_proba, load_unit_test, 0],
            ["BasicMotions", basic_motions_proba, load_basic_motions, 4],
        ]:
            # retrieve expected predict_proba output, and skip test if not available
            if classname in data_dict.keys():
                expected_probas = data_dict[classname]
            else:
                # skip test if no expected probas are registered
                continue

            # we only use the first estimator instance for testing
            estimator_instance = estimator_class.create_test_instance(
                parameter_set="results_comparison"
            )
            # set random seed if possible
            set_random_state(estimator_instance, 0)

            # load test data
            X_train, y_train = data_loader(split="train")
            X_test, _ = data_loader(split="test")
            indices = np.random.RandomState(data_seed).choice(
                len(y_train), 10, replace=False
            )

            # train classifier and predict probas
            estimator_instance.fit(X_train[indices], y_train[indices])
            y_proba, _ = estimator_instance.predict_proba(X_test[indices])

            # assert probabilities are the same
            _assert_array_almost_equal(
                y_proba,
                expected_probas,
                decimal=2,
                err_msg=f"Failed to reproduce results for {classname} on {data_name}",
            )
