"""Catch22 test code."""

import numpy as np
import pytest
from numpy import testing

from aeon.datasets import load_basic_motions
from aeon.transformations.collection.feature_based import Catch22
from aeon.utils.validation._dependencies import _check_soft_dependencies


def test_catch22_on_basic_motions():
    """Test of Catch22 on basic motions data."""
    X_train, _ = load_basic_motions(split="train")
    indices = [28, 39, 4, 15, 26]
    # fit Catch22 and assert transformed data is the same
    c22 = Catch22(replace_nans=True)
    data = c22.fit_transform(X_train[indices])
    testing.assert_array_almost_equal(data, catch22_basic_motions_data, decimal=4)

    # fit Catch22 with select features and assert transformed data is the same
    c22 = Catch22(replace_nans=True, features=feature_names)
    data = c22.fit_transform(X_train[indices])
    testing.assert_array_almost_equal(
        data,
        catch22_basic_motions_data[:, np.sort(np.r_[0:132:22, 2:132:22, 21:132:22])],
        decimal=4,
    )


def test_catch22_short_on_basic_motions():
    """Test of Catch22 on basic motions data."""
    X_train, _ = load_basic_motions(split="train")
    indices = [28, 39, 4, 15, 26]
    # fit Catch22 and assert transformed data is the same
    c22 = Catch22(features=feature_names_short, replace_nans=True)
    data = c22.fit_transform(X_train[indices])
    testing.assert_array_almost_equal(data, catch22_basic_motions_data, decimal=4)

    # fit Catch22 with select features and assert transformed data is the same
    c22 = Catch22(replace_nans=True, features=feature_names)
    data = c22.fit_transform(X_train[indices])
    testing.assert_array_almost_equal(
        data,
        catch22_basic_motions_data[:, np.sort(np.r_[0:132:22, 2:132:22, 21:132:22])],
        decimal=4,
    )


@pytest.mark.skipif(
    not _check_soft_dependencies("pycatch22", severity="none"),
    reason="skip test if required soft dependency pycatch22 not available",
)
def test_catch22_wrapper_on_basic_motions():
    """Test of Catch22Wrapper on basic motions data."""
    # load basic motions data
    X_train, _ = load_basic_motions(split="train")
    indices = np.random.RandomState(4).choice(len(X_train), 5, replace=False)

    # fit Catch22Wrapper and assert transformed data is the same
    c22 = Catch22(use_pycatch22=True, replace_nans=True)
    data = c22.fit_transform(X_train[indices])
    testing.assert_array_almost_equal(
        data, catch22wrapper_basic_motions_data, decimal=4
    )

    # fit Catch22Wrapper with select features and assert transformed data is the same
    c22 = Catch22(use_pycatch22=True, replace_nans=True, features=feature_names)
    data = c22.fit_transform(X_train[indices])
    testing.assert_array_almost_equal(
        data,
        catch22wrapper_basic_motions_data[
            :, np.sort(np.r_[0:132:22, 2:132:22, 21:132:22])
        ],
        decimal=4,
    )


feature_names = ["DN_HistogramMode_5", "CO_f1ecac", "FC_LocalSimple_mean3_stderr"]

feature_names_short = [
    "mode_5",
    "mode_10",
    "acf_timescale",
    "acf_first_min",
    "ami2",
    "trev",
    "high_fluctuation",
    "stretch_high",
    "transition_matrix",
    "periodicity",
    "embedding_dist",
    "ami_timescale",
    "whiten_timescale",
    "outlier_timing_pos",
    "outlier_timing_neg",
    "centroid_freq",
    "stretch_decreasing",
    "entropy_pairs",
    "rs_range",
    "dfa",
    "low_freq_power",
    "forecast_error",
]

# Both results should be the same
catch22_basic_motions_data = np.array(
    [
        [
            0.8918,
            1.2091,
            1.1428,
            3.0,
            0.0956,
            0.9846,
            0.9192,
            7.0,
            0.0108,
            13.0,
            0.0616,
            1.0,
            0.5,
            -0.2,
            0.04,
            0.4158,
            4.0,
            2.0909,
            0.3714,
            0.2,
            0.9327,
            1.4028,
            -1.8851,
            -3.4695,
            2.2361,
            6.0,
            0.2865,
            -0.9328,
            0.9192,
            8.0,
            0.0243,
            11.0,
            0.0934,
            2.0,
            0.75,
            -0.56,
            0.02,
            10.6256,
            8.0,
            1.8244,
            0.2857,
            0.1714,
            0.4909,
            3.4997,
            0.065,
            0.2902,
            1.5664,
            5.0,
            0.1343,
            0.1154,
            0.8485,
            9.0,
            0.0046,
            10.0,
            0.0967,
            2.0,
            0.3333,
            -0.78,
            -0.34,
            0.3406,
            8.0,
            1.9998,
            0.3143,
            0.6857,
            0.6381,
            0.9188,
            0.0216,
            0.0216,
            1.2329,
            7.0,
            0.1159,
            -0.0848,
            0.899,
            10.0,
            0.0101,
            12.0,
            0.1549,
            2.0,
            0.3333,
            -0.38,
            0.06,
            0.1938,
            5.0,
            2.0078,
            0.3143,
            0.1714,
            0.6381,
            0.712,
            0.3028,
            0.1962,
            1.815,
            7.0,
            0.1564,
            0.0199,
            0.8283,
            10.0,
            0.0046,
            13.0,
            0.5965,
            2.0,
            0.6667,
            0.03,
            0.35,
            0.1221,
            8.0,
            1.9443,
            0.2286,
            0.8,
            0.4909,
            0.4579,
            0.5644,
            0.9089,
            2.1736,
            6.0,
            0.1914,
            -0.4162,
            0.9091,
            8.0,
            0.0174,
            11.0,
            0.1072,
            2.0,
            1.0,
            -0.11,
            -0.72,
            1.7181,
            8.0,
            1.8142,
            0.2,
            0.7429,
            0.4909,
            1.4137,
        ],
        [
            0.1467,
            1.7614,
            1.851,
            8.0,
            0.1971,
            159.5298,
            0.9091,
            9.0,
            0.0091,
            3.0,
            0.1012,
            4.0,
            0.3333,
            -0.15,
            0.03,
            32.285,
            8.0,
            1.9501,
            0.6571,
            0.1714,
            0.4909,
            7.0913,
            -1.031,
            0.5692,
            1.3705,
            3.0,
            0.1008,
            -298.4341,
            0.9091,
            13.0,
            0.0037,
            8.0,
            0.0992,
            3.0,
            0.3333,
            0.2,
            -0.37,
            18.3029,
            7.0,
            2.0298,
            0.2,
            0.3143,
            0.7363,
            7.4417,
            0.1721,
            -1.3777,
            1.2531,
            3.0,
            0.156,
            91.0107,
            0.9091,
            12.0,
            0.0116,
            13.0,
            0.1303,
            3.0,
            0.3333,
            -0.23,
            -0.04,
            14.3938,
            5.0,
            2.0059,
            0.5429,
            0.7429,
            0.7854,
            6.5036,
            -0.2157,
            -1.7152,
            1.6441,
            5.0,
            0.0992,
            4.1153,
            0.899,
            13.0,
            0.0037,
            14.0,
            0.1047,
            2.0,
            0.3333,
            0.15,
            -0.18,
            7.1407,
            6.0,
            2.0097,
            0.2286,
            0.6571,
            0.6872,
            4.2472,
            0.6856,
            -0.1177,
            1.2534,
            4.0,
            0.0709,
            3.2192,
            0.8687,
            14.0,
            0.0073,
            6.0,
            0.064,
            1.0,
            0.3333,
            0.18,
            0.3,
            1.6007,
            5.0,
            2.0191,
            0.8286,
            0.7429,
            0.8345,
            3.0112,
            -1.5437,
            0.0972,
            1.1481,
            5.0,
            0.0292,
            119.5049,
            0.8889,
            8.0,
            0.0092,
            8.0,
            0.1307,
            2.0,
            0.3333,
            -0.14,
            0.1,
            7.3076,
            5.0,
            1.9736,
            0.2,
            0.2286,
            0.6872,
            5.194,
        ],
        [
            -0.2176,
            -0.252,
            0.9342,
            6.0,
            0.0569,
            -0.0018,
            0.697,
            7.0,
            0.0024,
            19.0,
            1.6321,
            2.0,
            0.3333,
            -0.13,
            0.02,
            0.0081,
            5.0,
            2.133,
            0.2,
            0.6,
            0.8836,
            0.1514,
            0.2537,
            0.3414,
            1.8247,
            5.0,
            0.213,
            0.0234,
            0.8586,
            8.0,
            0.0116,
            10.0,
            0.5715,
            2.0,
            1.0,
            -0.12,
            -0.02,
            0.1288,
            7.0,
            1.9505,
            0.2,
            0.8286,
            0.589,
            0.4569,
            -0.045,
            -0.0049,
            1.1595,
            3.0,
            0.0428,
            -0.0013,
            0.6162,
            9.0,
            0.0004,
            0.0,
            1.2419,
            1.0,
            1.0,
            -0.91,
            -0.22,
            0.0037,
            6.0,
            2.0869,
            0.1714,
            0.6,
            0.9327,
            0.1488,
            -0.0368,
            -0.0176,
            1.8915,
            5.0,
            0.1782,
            -0.0,
            0.3939,
            11.0,
            0.0122,
            0.0,
            3.976,
            2.0,
            0.25,
            -0.39,
            -0.08,
            0.0041,
            7.0,
            1.9728,
            0.2571,
            0.7429,
            0.589,
            0.0774,
            0.02,
            0.0044,
            1.7819,
            5.0,
            0.0615,
            -0.0,
            0.2424,
            8.0,
            0.0043,
            0.0,
            6.8497,
            2.0,
            0.3333,
            -0.06,
            0.05,
            0.0013,
            7.0,
            2.039,
            0.2,
            0.7429,
            0.589,
            0.0469,
            -0.0914,
            -0.1258,
            1.9983,
            5.0,
            0.1025,
            0.0001,
            0.7071,
            11.0,
            0.0037,
            10.0,
            3.1416,
            2.0,
            1.0,
            -0.155,
            0.125,
            0.0212,
            7.0,
            1.8706,
            0.2,
            0.8,
            0.589,
            0.1728,
        ],
        [
            14.753,
            12.6115,
            1.3139,
            5.0,
            0.2837,
            611.2311,
            0.8485,
            8.0,
            0.0046,
            7.0,
            0.1723,
            1.0,
            1.0,
            -0.01,
            -0.17,
            8.3186,
            5.0,
            2.0538,
            0.2286,
            0.2571,
            0.7363,
            15.2468,
            -8.7478,
            -11.0416,
            1.2448,
            4.0,
            0.2358,
            -666.9228,
            0.8586,
            5.0,
            0.005,
            7.0,
            0.1222,
            1.0,
            1.0,
            0.09,
            0.01,
            5.3016,
            4.0,
            2.0075,
            0.1714,
            0.8286,
            0.7363,
            16.1192,
            -1.1495,
            -3.2478,
            0.8858,
            2.0,
            0.095,
            3.9326,
            0.8586,
            8.0,
            0.0012,
            3.0,
            0.0841,
            2.0,
            0.5,
            0.13,
            -0.08,
            1.7627,
            5.0,
            2.1476,
            0.7714,
            0.6,
            1.4726,
            3.3443,
            0.0945,
            -1.7292,
            0.9238,
            3.0,
            0.1318,
            13.9244,
            0.8586,
            5.0,
            0.0124,
            8.0,
            0.0608,
            1.0,
            0.5,
            0.06,
            0.12,
            0.5924,
            7.0,
            2.1388,
            0.2286,
            0.8286,
            0.8836,
            2.8217,
            -0.2413,
            -0.7284,
            1.124,
            4.0,
            0.1682,
            -0.314,
            0.8384,
            6.0,
            0.0165,
            8.0,
            0.0498,
            1.0,
            0.5,
            -0.05,
            -0.11,
            0.2086,
            6.0,
            2.0597,
            0.1714,
            0.8286,
            0.7363,
            2.9013,
            -0.2211,
            0.9037,
            1.473,
            4.0,
            0.2979,
            -63.8967,
            0.8586,
            6.0,
            0.0165,
            8.0,
            0.0848,
            1.0,
            0.6667,
            0.01,
            0.04,
            1.3957,
            5.0,
            1.9356,
            0.1714,
            0.8286,
            0.7363,
            7.0449,
        ],
        [
            -0.0619,
            0.1991,
            1.2578,
            4.0,
            0.1921,
            1.2447,
            0.899,
            6.0,
            0.0089,
            6.0,
            0.0718,
            1.0,
            0.3333,
            0.03,
            0.13,
            0.501,
            6.0,
            2.0492,
            0.2,
            0.1714,
            0.8836,
            1.5288,
            -3.0176,
            -3.5235,
            2.5291,
            7.0,
            0.3019,
            -1.2227,
            0.9091,
            9.0,
            0.0156,
            13.0,
            0.099,
            2.0,
            1.0,
            0.0,
            0.1,
            7.6385,
            8.0,
            1.7807,
            0.3143,
            0.1714,
            0.4418,
            2.509,
            -0.519,
            -0.6649,
            2.0582,
            6.0,
            0.2148,
            -0.0086,
            0.8384,
            14.0,
            0.0122,
            13.0,
            0.4267,
            2.0,
            0.5,
            -0.13,
            0.19,
            0.3096,
            6.0,
            1.8881,
            0.2,
            0.6571,
            0.4418,
            0.669,
            -0.2996,
            -0.0955,
            2.0035,
            8.0,
            0.2115,
            -0.1388,
            0.8384,
            8.0,
            0.0069,
            14.0,
            0.2553,
            2.0,
            0.5,
            0.18,
            0.22,
            0.379,
            7.0,
            1.9223,
            0.2857,
            0.1714,
            0.4418,
            0.7113,
            -0.3873,
            0.4293,
            2.5648,
            7.0,
            0.2377,
            0.0065,
            0.8081,
            9.0,
            0.0122,
            13.0,
            0.6829,
            3.0,
            1.0,
            0.11,
            0.35,
            0.2719,
            7.0,
            1.7647,
            0.2571,
            0.1714,
            0.4418,
            0.4799,
            -0.9868,
            -1.4515,
            2.5809,
            7.0,
            0.2511,
            -0.0475,
            0.8687,
            9.0,
            0.0208,
            13.0,
            0.343,
            3.0,
            1.0,
            -0.04,
            0.08,
            1.5658,
            8.0,
            1.7683,
            0.2857,
            0.1714,
            0.4418,
            1.1302,
        ],
    ]
)

catch22wrapper_basic_motions_data = np.array(
    [
        [
            0.0804,
            0.3578,
            1.1428,
            3.0,
            0.0956,
            0.6572,
            0.9192,
            7.0,
            0.0108,
            13.0,
            0.0653,
            1.0,
            0.5,
            -0.28,
            0.04,
            0.3176,
            4.0,
            2.0909,
            0.3714,
            0.2,
            0.9327,
            1.226,
            -0.5321,
            -0.9703,
            2.2361,
            6.0,
            0.281,
            -0.0197,
            0.8889,
            8.0,
            0.0243,
            11.0,
            0.3456,
            2.0,
            0.75,
            -0.05,
            0.02,
            0.8128,
            8.0,
            1.8244,
            0.2857,
            0.1714,
            0.4909,
            0.968,
            0.2931,
            0.5641,
            1.5664,
            5.0,
            0.1343,
            0.2011,
            0.8485,
            9.0,
            0.0046,
            10.0,
            0.0597,
            2.0,
            0.3333,
            -0.78,
            -0.34,
            0.4932,
            8.0,
            1.9998,
            0.3143,
            0.6857,
            0.6381,
            1.1057,
            0.0448,
            0.0448,
            1.2329,
            7.0,
            0.1159,
            -0.3243,
            0.899,
            10.0,
            0.0101,
            12.0,
            0.0806,
            2.0,
            0.3333,
            -0.38,
            0.06,
            0.474,
            5.0,
            2.0078,
            0.3143,
            0.1714,
            0.6381,
            1.1134,
            0.7638,
            0.5217,
            1.815,
            7.0,
            0.13,
            0.2326,
            0.899,
            10.0,
            0.0046,
            13.0,
            0.1696,
            2.0,
            0.6667,
            0.03,
            0.35,
            0.6291,
            8.0,
            1.9443,
            0.2286,
            0.8,
            0.4909,
            1.0392,
            0.4149,
            0.6463,
            2.1736,
            6.0,
            0.185,
            -0.1261,
            0.9091,
            8.0,
            0.0174,
            11.0,
            0.2083,
            2.0,
            1.0,
            -0.11,
            -0.805,
            0.7752,
            8.0,
            1.8142,
            0.2,
            0.7429,
            0.4909,
            0.9495,
        ],
        [
            -0.8062,
            -0.5798,
            1.851,
            8.0,
            0.1823,
            0.44,
            0.8586,
            9.0,
            0.0091,
            24.0,
            0.1544,
            4.0,
            0.3333,
            -0.16,
            0.01,
            0.635,
            8.0,
            1.9501,
            0.6571,
            0.1714,
            0.4909,
            0.9945,
            -0.1266,
            0.1157,
            1.3705,
            3.0,
            0.1041,
            -1.0354,
            0.8889,
            13.0,
            0.0037,
            8.0,
            0.0602,
            3.0,
            0.3333,
            0.2,
            -0.37,
            0.4195,
            7.0,
            2.0298,
            0.2,
            0.3143,
            0.7363,
            1.1266,
            0.4734,
            0.207,
            1.2531,
            3.0,
            0.1742,
            0.4621,
            0.8687,
            12.0,
            0.0116,
            13.0,
            0.0523,
            3.0,
            0.3333,
            -0.23,
            0.06,
            0.4252,
            5.0,
            2.0059,
            0.5429,
            0.7429,
            0.7854,
            1.1178,
            -0.0013,
            -0.3798,
            1.6441,
            5.0,
            0.0992,
            0.0662,
            0.8788,
            13.0,
            0.0037,
            14.0,
            0.0879,
            2.0,
            0.3333,
            0.06,
            -0.2,
            0.4549,
            6.0,
            2.0097,
            0.2286,
            0.6571,
            0.6872,
            1.0719,
            0.2664,
            -0.0514,
            1.2534,
            4.0,
            0.0692,
            0.1995,
            0.8081,
            14.0,
            0.0073,
            6.0,
            0.0673,
            1.0,
            0.3333,
            0.19,
            0.3,
            0.2506,
            5.0,
            2.0191,
            0.8286,
            0.7429,
            0.8345,
            1.1916,
            -0.3504,
            0.0039,
            1.1481,
            5.0,
            0.0276,
            1.2034,
            0.7778,
            8.0,
            0.0092,
            8.0,
            0.0342,
            2.0,
            0.3333,
            -0.14,
            -0.04,
            0.3408,
            5.0,
            1.9736,
            0.2,
            0.2286,
            0.6872,
            1.1216,
        ],
        [
            0.3645,
            0.1094,
            0.9342,
            6.0,
            0.0879,
            -0.7506,
            0.8687,
            7.0,
            0.0024,
            7.0,
            0.0845,
            2.0,
            0.3333,
            -0.13,
            -0.34,
            0.4448,
            5.0,
            2.133,
            0.2,
            0.6,
            0.8836,
            1.1237,
            0.3658,
            0.5744,
            1.8247,
            5.0,
            0.2149,
            0.3158,
            0.8687,
            8.0,
            0.0116,
            10.0,
            0.1758,
            2.0,
            1.0,
            -0.23,
            -0.01,
            0.7293,
            7.0,
            1.9505,
            0.2,
            0.8286,
            0.589,
            1.087,
            -0.3643,
            -0.0421,
            1.1595,
            3.0,
            0.13,
            -0.6767,
            0.8182,
            9.0,
            0.0004,
            7.0,
            0.0611,
            1.0,
            1.0,
            -0.91,
            -0.22,
            0.2371,
            6.0,
            2.0869,
            0.1714,
            0.6,
            0.9327,
            1.1946,
            -0.3788,
            -0.1279,
            1.8915,
            5.0,
            0.1358,
            -0.0644,
            0.8586,
            11.0,
            0.0122,
            10.0,
            0.1512,
            2.0,
            0.25,
            -0.39,
            -0.08,
            0.6975,
            7.0,
            1.9728,
            0.2571,
            0.7429,
            0.589,
            1.0127,
            0.2208,
            -0.1147,
            1.7819,
            5.0,
            0.0862,
            -0.1311,
            0.8687,
            8.0,
            0.0043,
            12.0,
            0.1591,
            2.0,
            0.3333,
            -0.08,
            0.03,
            0.6061,
            7.0,
            2.039,
            0.2,
            0.7429,
            0.589,
            1.0105,
            -0.6362,
            -0.8457,
            1.9983,
            5.0,
            0.1284,
            0.0228,
            0.8586,
            11.0,
            0.0037,
            10.0,
            0.3778,
            2.0,
            1.0,
            -0.2,
            0.12,
            0.7828,
            7.0,
            1.8706,
            0.2,
            0.8,
            0.589,
            1.0502,
        ],
        [
            0.7285,
            0.5526,
            1.3139,
            5.0,
            0.254,
            0.3383,
            0.8081,
            8.0,
            0.0046,
            7.0,
            0.1392,
            1.0,
            1.0,
            -0.07,
            -0.17,
            0.0561,
            5.0,
            2.0538,
            0.2286,
            0.2571,
            0.7363,
            1.2518,
            -0.3475,
            -0.5286,
            1.2448,
            4.0,
            0.2395,
            -0.3284,
            0.8283,
            5.0,
            0.005,
            7.0,
            0.1219,
            1.0,
            1.0,
            0.08,
            -0.01,
            0.0331,
            4.0,
            2.0075,
            0.1714,
            0.8286,
            0.7363,
            1.2729,
            0.4846,
            -0.2633,
            0.8858,
            2.0,
            0.1024,
            0.1781,
            0.8485,
            8.0,
            0.0012,
            3.0,
            0.0817,
            2.0,
            0.5,
            -0.02,
            -0.12,
            0.2239,
            5.0,
            2.1476,
            0.7714,
            0.6,
            1.4726,
            1.192,
            0.0544,
            -0.747,
            0.9238,
            3.0,
            0.1503,
            1.1815,
            0.8283,
            5.0,
            0.0124,
            8.0,
            0.0664,
            1.0,
            0.5,
            0.06,
            0.12,
            0.1144,
            7.0,
            2.1388,
            0.2286,
            0.8286,
            0.8836,
            1.2399,
            -0.1774,
            -0.3877,
            1.124,
            4.0,
            0.1556,
            -0.0252,
            0.8182,
            6.0,
            0.0165,
            8.0,
            0.086,
            1.0,
            0.5,
            -0.05,
            -0.1,
            0.0388,
            6.0,
            2.0597,
            0.1714,
            0.8286,
            0.7363,
            1.2521,
            -0.0394,
            0.1582,
            1.473,
            4.0,
            0.3,
            -0.3468,
            0.8586,
            6.0,
            0.0165,
            8.0,
            0.2129,
            1.0,
            0.6667,
            0.01,
            0.04,
            0.0431,
            5.0,
            1.9356,
            0.1714,
            0.8286,
            0.7363,
            1.2381,
        ],
        [
            -0.6177,
            -0.4164,
            1.2578,
            4.0,
            0.1921,
            0.5714,
            0.899,
            6.0,
            0.0089,
            6.0,
            0.1321,
            1.0,
            0.3333,
            0.03,
            -0.14,
            0.2982,
            6.0,
            2.0492,
            0.2,
            0.1714,
            0.8836,
            1.1794,
            -1.0401,
            -1.2153,
            2.5291,
            7.0,
            0.3019,
            -0.0508,
            0.8889,
            9.0,
            0.0156,
            13.0,
            0.3581,
            2.0,
            1.0,
            0.0,
            0.1,
            0.9158,
            8.0,
            1.7807,
            0.3143,
            0.1714,
            0.4418,
            0.8688,
            -0.2369,
            -0.4506,
            2.0582,
            6.0,
            0.2042,
            -0.0271,
            0.8586,
            14.0,
            0.0122,
            13.0,
            0.2277,
            2.0,
            0.5,
            -0.13,
            0.3,
            0.665,
            6.0,
            1.8881,
            0.2,
            0.6571,
            0.4418,
            0.9804,
            -0.4152,
            -0.1326,
            2.0035,
            8.0,
            0.1768,
            -0.3678,
            0.8384,
            8.0,
            0.0069,
            14.0,
            0.1498,
            2.0,
            0.5,
            0.18,
            0.22,
            0.7259,
            7.0,
            1.9223,
            0.2857,
            0.1714,
            0.4418,
            0.9844,
            -0.7112,
            0.7554,
            2.5648,
            7.0,
            0.2621,
            0.0377,
            0.8283,
            9.0,
            0.0122,
            13.0,
            0.317,
            3.0,
            1.0,
            0.11,
            0.22,
            0.8769,
            7.0,
            1.7647,
            0.2571,
            0.1714,
            0.4418,
            0.8619,
            -0.7754,
            -1.133,
            2.5809,
            7.0,
            0.2648,
            -0.0217,
            0.8485,
            9.0,
            0.0208,
            13.0,
            0.4885,
            3.0,
            1.0,
            -0.04,
            0.08,
            0.9274,
            8.0,
            1.7683,
            0.2857,
            0.1714,
            0.4418,
            0.8698,
        ],
    ]
)
