#include "vk/vulkan_core.h"
#include <threadContext.hpp>
#include <util/util.hpp>
#include <util/bytes.hpp>
#include <util/f16.hpp>
#include <util/vecOps.hpp>
#include <util/dlg.hpp>
#include <vk/typemap_helper.h>
#include <vk/vk_layer.h>
#include <vk/format_utils.h>
#include <vk/enumString.hpp>
#include <cmath>
#include <cstdio>

namespace vil {

// high-level stuff
u32 findLSB(u32 v) {
	// https://stackoverflow.com/questions/757059
	static const int blackMagic[32] = {
		0, 1, 28, 2, 29, 14, 24, 3, 30, 22, 20, 15, 25, 17, 4, 8,
		31, 27, 13, 23, 21, 19, 16, 7, 26, 12, 18, 6, 11, 5, 10, 9
	};
	return blackMagic[((u32)((v & (~v + 1)) * 0x077CB531U)) >> 27];
}

u32 nextPOT(u32 v) {
    dlg_assert(v > 0);

	--v;
	v |= v >> 1;
	v |= v >> 2;
	v |= v >> 4;
	v |= v >> 8;
	v |= v >> 16;
	++v;

    return v;
}

bool checkEnvBinary(const char* env, bool defaultValue) {
	auto e = std::getenv(env);
	if(!e) {
		return defaultValue;
	}

	if(std::strcmp(e, "0") == 0u) {
		return false;
	}

	if(std::strcmp(e, "1") == 0u) {
		return true;
	}

	dlg_warn("Environment variable '{}' set to invalid value '{}'. Expected 0 or 1",
		env, e);
	return defaultValue;
}

VkImageType minImageType(VkExtent3D size, unsigned minDim) {
	if(size.depth > 1 || minDim > 2) {
		return VK_IMAGE_TYPE_3D;
	} else if(size.height > 1 || minDim > 1) {
		return VK_IMAGE_TYPE_2D;
	} else {
		return VK_IMAGE_TYPE_1D;
	}
}

VkImageViewType minImageViewType(VkExtent3D size, unsigned layers,
		bool cubemap, unsigned minDim) {
	if(size.depth > 1 || minDim > 2) {
		dlg_assertm(layers <= 1 && cubemap == 0,
			"Layered or cube 3D images are not allowed");
		return VK_IMAGE_VIEW_TYPE_3D;
	}

	if(cubemap) {
		dlg_assert(layers % 6 == 0u);
		return (layers > 6 ? VK_IMAGE_VIEW_TYPE_CUBE_ARRAY : VK_IMAGE_VIEW_TYPE_CUBE);
	}

	if(size.height > 1 || minDim > 1) {
		return layers > 1 ? VK_IMAGE_VIEW_TYPE_2D_ARRAY : VK_IMAGE_VIEW_TYPE_2D;
	} else {
		return layers > 1 ? VK_IMAGE_VIEW_TYPE_1D_ARRAY : VK_IMAGE_VIEW_TYPE_1D;
	}
}

std::size_t structSize(VkStructureType type) {
	static const std::unordered_map<VkStructureType, std::size_t> structSizes = {
#define ENTRY(stype) {stype, sizeof(LvlSTypeMap<stype>::Type)}
    	ENTRY(VK_STRUCTURE_TYPE_APPLICATION_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_SUBMIT_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE),
    	ENTRY(VK_STRUCTURE_TYPE_BIND_SPARSE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_FENCE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_EVENT_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_BUFFER_VIEW_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_SHADER_MODULE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_CACHE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_INPUT_ASSEMBLY_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_MULTISAMPLE_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_DEPTH_STENCIL_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_DYNAMIC_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_SET_ALLOCATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET),
    	ENTRY(VK_STRUCTURE_TYPE_COPY_DESCRIPTOR_SET),
    	ENTRY(VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_BARRIER),
		{VK_STRUCTURE_TYPE_LOADER_INSTANCE_CREATE_INFO, sizeof(VkLayerInstanceCreateInfo)},
		{VK_STRUCTURE_TYPE_LOADER_DEVICE_CREATE_INFO, sizeof(VkLayerDeviceCreateInfo)},
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2),
    	ENTRY(VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2),
    	ENTRY(VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2),
    	ENTRY(VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2),
    	ENTRY(VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTERS_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PROTECTED_SUBMIT_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_QUEUE_INFO_2),
    	ENTRY(VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DRAW_PARAMETERS_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_1_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_1_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_2_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_1_2_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_FORMAT_LIST_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_2),
    	ENTRY(VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_2),
    	ENTRY(VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_2),
    	ENTRY(VK_STRUCTURE_TYPE_SUBPASS_DEPENDENCY_2),
    	ENTRY(VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO_2),
    	ENTRY(VK_STRUCTURE_TYPE_SUBPASS_BEGIN_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_SUBPASS_END_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_8BIT_STORAGE_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRIVER_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_INT64_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_FLOAT16_INT8_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT_CONTROLS_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_BINDING_FLAGS_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_ALLOCATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_LAYOUT_SUPPORT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_STENCIL_RESOLVE_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_DEPTH_STENCIL_RESOLVE),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCALAR_BLOCK_LAYOUT_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_STENCIL_USAGE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_FILTER_MINMAX_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_SAMPLER_REDUCTION_MODE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_MEMORY_MODEL_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGELESS_FRAMEBUFFER_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_FRAMEBUFFER_ATTACHMENTS_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_FRAMEBUFFER_ATTACHMENT_IMAGE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_RENDER_PASS_ATTACHMENT_BEGIN_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_UNIFORM_BUFFER_STANDARD_LAYOUT_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SUBGROUP_EXTENDED_TYPES_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SEPARATE_DEPTH_STENCIL_LAYOUTS_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_STENCIL_LAYOUT),
    	ENTRY(VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_STENCIL_LAYOUT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_HOST_QUERY_RESET_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TIMELINE_SEMAPHORE_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TIMELINE_SEMAPHORE_PROPERTIES),
    	ENTRY(VK_STRUCTURE_TYPE_SEMAPHORE_TYPE_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_TIMELINE_SEMAPHORE_SUBMIT_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_SEMAPHORE_WAIT_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_SEMAPHORE_SIGNAL_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_DEVICE_ADDRESS_FEATURES),
    	ENTRY(VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_BUFFER_OPAQUE_CAPTURE_ADDRESS_CREATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_OPAQUE_CAPTURE_ADDRESS_ALLOCATE_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_MEMORY_OPAQUE_CAPTURE_ADDRESS_INFO),
    	ENTRY(VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PRESENT_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_CAPABILITIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_SWAPCHAIN_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_SWAPCHAIN_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACQUIRE_NEXT_IMAGE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_GROUP_SWAPCHAIN_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_MODE_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_SURFACE_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_PRESENT_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_XLIB_SURFACE_CREATE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_XCB_SURFACE_CREATE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_WAYLAND_SURFACE_CREATE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_ANDROID_SURFACE_CREATE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_RASTERIZATION_ORDER_AMD),
    	ENTRY(VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_NAME_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_TAG_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEBUG_MARKER_MARKER_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_IMAGE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_BUFFER_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_MEMORY_ALLOCATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_VIEW_HANDLE_INFO_NVX),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_VIEW_ADDRESS_PROPERTIES_NVX),
    	ENTRY(VK_STRUCTURE_TYPE_TEXTURE_LOD_GATHER_FORMAT_PROPERTIES_AMD),
    	// ENTRY(VK_STRUCTURE_TYPE_STREAM_DESCRIPTOR_SURFACE_CREATE_INFO_GGP),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CORNER_SAMPLED_IMAGE_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_NV),
    	// ENTRY(VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_NV),
    	// ENTRY(VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_NV),
    	// ENTRY(VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_VALIDATION_FLAGS_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_VI_SURFACE_CREATE_INFO_NN),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXTURE_COMPRESSION_ASTC_HDR_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_VIEW_ASTC_DECODE_MODE_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ASTC_DECODE_FEATURES_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_MEMORY_WIN32_HANDLE_PROPERTIES_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_MEMORY_GET_WIN32_HANDLE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_IMPORT_MEMORY_FD_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_FD_PROPERTIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_GET_FD_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_D3D12_FENCE_SUBMIT_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_SEMAPHORE_GET_WIN32_HANDLE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_FD_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_SEMAPHORE_GET_FD_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PUSH_DESCRIPTOR_PROPERTIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_CONDITIONAL_RENDERING_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONDITIONAL_RENDERING_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_CONDITIONAL_RENDERING_BEGIN_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PRESENT_REGIONS_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_W_SCALING_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_POWER_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_EVENT_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_EVENT_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_SWAPCHAIN_COUNTER_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PRESENT_TIMES_INFO_GOOGLE),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_ATTRIBUTES_PROPERTIES_NVX),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SWIZZLE_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISCARD_RECTANGLE_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_DISCARD_RECTANGLE_STATE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONSERVATIVE_RASTERIZATION_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_CONSERVATIVE_STATE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_CLIP_ENABLE_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_DEPTH_CLIP_STATE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_HDR_METADATA_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_SHARED_PRESENT_SURFACE_CAPABILITIES_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_IMPORT_FENCE_WIN32_HANDLE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_EXPORT_FENCE_WIN32_HANDLE_INFO_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_FENCE_GET_WIN32_HANDLE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_IMPORT_FENCE_FD_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_FENCE_GET_FD_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PERFORMANCE_QUERY_FEATURES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PERFORMANCE_QUERY_PROPERTIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_QUERY_POOL_PERFORMANCE_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PERFORMANCE_QUERY_SUBMIT_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACQUIRE_PROFILING_LOCK_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PERFORMANCE_COUNTER_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PERFORMANCE_COUNTER_DESCRIPTION_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SURFACE_INFO_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_SURFACE_FORMAT_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_PROPERTIES_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_PLANE_PROPERTIES_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_MODE_PROPERTIES_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_PLANE_INFO_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_PLANE_CAPABILITIES_2_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_IOS_SURFACE_CREATE_INFO_MVK),
    	// ENTRY(VK_STRUCTURE_TYPE_MACOS_SURFACE_CREATE_INFO_MVK),
    	ENTRY(VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_NAME_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_TAG_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEBUG_UTILS_LABEL_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CALLBACK_DATA_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_USAGE_ANDROID),
    	// ENTRY(VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_PROPERTIES_ANDROID),
    	// ENTRY(VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_PROPERTIES_ANDROID),
    	// ENTRY(VK_STRUCTURE_TYPE_IMPORT_ANDROID_HARDWARE_BUFFER_INFO_ANDROID),
    	// ENTRY(VK_STRUCTURE_TYPE_MEMORY_GET_ANDROID_HARDWARE_BUFFER_INFO_ANDROID),
    	// ENTRY(VK_STRUCTURE_TYPE_EXTERNAL_FORMAT_ANDROID),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_INLINE_UNIFORM_BLOCK_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_INLINE_UNIFORM_BLOCK_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_SAMPLE_LOCATIONS_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_RENDER_PASS_SAMPLE_LOCATIONS_BEGIN_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_SAMPLE_LOCATIONS_STATE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLE_LOCATIONS_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_MULTISAMPLE_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_ADVANCED_STATE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_TO_COLOR_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_BUILD_GEOMETRY_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_DEVICE_ADDRESS_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_AABBS_DATA_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_INSTANCES_DATA_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_TRIANGLES_DATA_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_GEOMETRY_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_VERSION_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_COPY_ACCELERATION_STRUCTURE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_COPY_ACCELERATION_STRUCTURE_TO_MEMORY_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_COPY_MEMORY_TO_ACCELERATION_STRUCTURE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ACCELERATION_STRUCTURE_FEATURES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ACCELERATION_STRUCTURE_PROPERTIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_BUILD_SIZES_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PIPELINE_FEATURES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PIPELINE_PROPERTIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_INTERFACE_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_QUERY_FEATURES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_MODULATION_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SM_BUILTINS_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_SM_BUILTINS_PROPERTIES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_LIST_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_DRM_FORMAT_MODIFIER_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_LIST_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_EXPLICIT_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_VALIDATION_CACHE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_SHADER_MODULE_VALIDATION_CACHE_CREATE_INFO_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PORTABILITY_SUBSET_FEATURES_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PORTABILITY_SUBSET_PROPERTIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SHADING_RATE_IMAGE_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_PROPERTIES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_COARSE_SAMPLE_ORDER_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_GEOMETRY_NV),
    	ENTRY(VK_STRUCTURE_TYPE_GEOMETRY_TRIANGLES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_GEOMETRY_AABB_NV),
    	ENTRY(VK_STRUCTURE_TYPE_BIND_ACCELERATION_STRUCTURE_MEMORY_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_NV),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PROPERTIES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_REPRESENTATIVE_FRAGMENT_TEST_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_REPRESENTATIVE_FRAGMENT_TEST_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_VIEW_IMAGE_FORMAT_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_FILTER_CUBIC_IMAGE_VIEW_IMAGE_FORMAT_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_QUEUE_GLOBAL_PRIORITY_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_IMPORT_MEMORY_HOST_POINTER_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_HOST_POINTER_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_HOST_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CLOCK_FEATURES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_COMPILER_CONTROL_CREATE_INFO_AMD),
    	ENTRY(VK_STRUCTURE_TYPE_CALIBRATED_TIMESTAMP_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_AMD),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_MEMORY_OVERALLOCATION_CREATE_INFO_AMD),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_DIVISOR_STATE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_FEATURES_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_PRESENT_FRAME_TOKEN_GGP),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_CREATION_FEEDBACK_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COMPUTE_SHADER_DERIVATIVES_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_PROPERTIES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_FOOTPRINT_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_EXCLUSIVE_SCISSOR_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXCLUSIVE_SCISSOR_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_CHECKPOINT_DATA_NV),
    	ENTRY(VK_STRUCTURE_TYPE_QUEUE_FAMILY_CHECKPOINT_PROPERTIES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_FUNCTIONS_2_FEATURES_INTEL),
    	ENTRY(VK_STRUCTURE_TYPE_QUERY_POOL_PERFORMANCE_QUERY_CREATE_INFO_INTEL),
    	ENTRY(VK_STRUCTURE_TYPE_INITIALIZE_PERFORMANCE_API_INFO_INTEL),
    	ENTRY(VK_STRUCTURE_TYPE_PERFORMANCE_MARKER_INFO_INTEL),
    	ENTRY(VK_STRUCTURE_TYPE_PERFORMANCE_STREAM_MARKER_INFO_INTEL),
    	ENTRY(VK_STRUCTURE_TYPE_PERFORMANCE_OVERRIDE_INFO_INTEL),
    	ENTRY(VK_STRUCTURE_TYPE_PERFORMANCE_CONFIGURATION_ACQUIRE_INFO_INTEL),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PCI_BUS_INFO_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DISPLAY_NATIVE_HDR_SURFACE_CAPABILITIES_AMD),
    	ENTRY(VK_STRUCTURE_TYPE_SWAPCHAIN_DISPLAY_NATIVE_HDR_CREATE_INFO_AMD),
    	// ENTRY(VK_STRUCTURE_TYPE_IMAGEPIPE_SURFACE_CREATE_INFO_FUCHSIA),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_TERMINATE_INVOCATION_FEATURES_KHR),
    	// ENTRY(VK_STRUCTURE_TYPE_METAL_SURFACE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_RENDER_PASS_FRAGMENT_DENSITY_MAP_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_REQUIRED_SUBGROUP_SIZE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_FRAGMENT_SHADING_RATE_ATTACHMENT_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_FRAGMENT_SHADING_RATE_STATE_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_PROPERTIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_FEATURES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_2_AMD),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COHERENT_MEMORY_FEATURES_AMD),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_ATOMIC_INT64_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_BUDGET_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PRIORITY_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_PRIORITY_ALLOCATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_SURFACE_PROTECTED_CAPABILITIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEDICATED_ALLOCATION_IMAGE_ALIASING_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_DEVICE_ADDRESS_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TOOL_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_VALIDATION_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_COOPERATIVE_MATRIX_PROPERTIES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_PROPERTIES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COVERAGE_REDUCTION_MODE_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_REDUCTION_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_FRAMEBUFFER_MIXED_SAMPLES_COMBINATION_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_INTERLOCK_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_YCBCR_IMAGE_ARRAYS_FEATURES_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_SURFACE_FULL_SCREEN_EXCLUSIVE_INFO_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_FULL_SCREEN_EXCLUSIVE_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_SURFACE_FULL_SCREEN_EXCLUSIVE_WIN32_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_HEADLESS_SURFACE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LINE_RASTERIZATION_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_LINE_STATE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LINE_RASTERIZATION_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_FLOAT_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INDEX_TYPE_UINT8_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_EXECUTABLE_PROPERTIES_FEATURES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_PROPERTIES_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_STATISTIC_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_EXECUTABLE_INTERNAL_REPRESENTATION_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DEMOTE_TO_HELPER_INVOCATION_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_GENERATED_COMMANDS_PROPERTIES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_GRAPHICS_SHADER_GROUP_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_SHADER_GROUPS_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_TOKEN_NV),
    	ENTRY(VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_GENERATED_COMMANDS_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_GENERATED_COMMANDS_MEMORY_REQUIREMENTS_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_GENERATED_COMMANDS_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXEL_BUFFER_ALIGNMENT_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXEL_BUFFER_ALIGNMENT_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_RENDER_PASS_TRANSFORM_INFO_QCOM),
    	ENTRY(VK_STRUCTURE_TYPE_RENDER_PASS_TRANSFORM_BEGIN_INFO_QCOM),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEVICE_MEMORY_REPORT_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_DEVICE_MEMORY_REPORT_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_MEMORY_REPORT_CALLBACK_DATA_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ROBUSTNESS_2_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ROBUSTNESS_2_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_SAMPLER_CUSTOM_BORDER_COLOR_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUSTOM_BORDER_COLOR_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CUSTOM_BORDER_COLOR_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_LIBRARY_CREATE_INFO_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRIVATE_DATA_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_PRIVATE_DATA_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PRIVATE_DATA_SLOT_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PIPELINE_CREATION_CACHE_CONTROL_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DIAGNOSTICS_CONFIG_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_DEVICE_DIAGNOSTICS_CONFIG_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_ENUMS_PROPERTIES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADING_RATE_ENUMS_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_FRAGMENT_SHADING_RATE_ENUM_STATE_CREATE_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_2_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_2_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_COPY_COMMAND_TRANSFORM_INFO_QCOM),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_ROBUSTNESS_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_COPY_BUFFER_INFO_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_COPY_IMAGE_INFO_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_COPY_BUFFER_TO_IMAGE_INFO_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_COPY_IMAGE_TO_BUFFER_INFO_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_BLIT_IMAGE_INFO_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_RESOLVE_IMAGE_INFO_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_BUFFER_COPY_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_COPY_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_BLIT_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_BUFFER_IMAGE_COPY_2_KHR),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_RESOLVE_2_KHR),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_PROPERTIES),
		ENTRY(VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_REQUIRED_SUBGROUP_SIZE_CREATE_INFO),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_SIZE_CONTROL_FEATURES),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_FEATURES),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_PROPERTIES),
		ENTRY(VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_INLINE_UNIFORM_BLOCK),
		ENTRY(VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_INLINE_UNIFORM_BLOCK_CREATE_INFO),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXTURE_COMPRESSION_ASTC_HDR_FEATURES),
		ENTRY(VK_STRUCTURE_TYPE_RENDERING_INFO),
		ENTRY(VK_STRUCTURE_TYPE_RENDERING_ATTACHMENT_INFO),
		ENTRY(VK_STRUCTURE_TYPE_PIPELINE_RENDERING_CREATE_INFO),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DYNAMIC_RENDERING_FEATURES),
		ENTRY(VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_RENDERING_INFO),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_DOT_PRODUCT_FEATURES),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_INTEGER_DOT_PRODUCT_PROPERTIES),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TEXEL_BUFFER_ALIGNMENT_PROPERTIES),
		ENTRY(VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_3),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_4_FEATURES),
		ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_4_PROPERTIES),
		ENTRY(VK_STRUCTURE_TYPE_DEVICE_BUFFER_MEMORY_REQUIREMENTS),
		ENTRY(VK_STRUCTURE_TYPE_DEVICE_IMAGE_MEMORY_REQUIREMENTS),

    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_4444_FORMATS_FEATURES_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_DIRECTFB_SURFACE_CREATE_INFO_EXT),

    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RASTERIZATION_ORDER_ATTACHMENT_ACCESS_FEATURES_ARM),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RGBA10X6_FORMATS_FEATURES_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_DIRECTFB_SURFACE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MUTABLE_DESCRIPTOR_TYPE_FEATURES_VALVE),
    	ENTRY(VK_STRUCTURE_TYPE_MUTABLE_DESCRIPTOR_TYPE_CREATE_INFO_VALVE),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_INPUT_DYNAMIC_STATE_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_VERTEX_INPUT_BINDING_DESCRIPTION_2_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_VERTEX_INPUT_ATTRIBUTE_DESCRIPTION_2_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRM_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_CLIP_CONTROL_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_DEPTH_CLIP_CONTROL_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PRIMITIVE_TOPOLOGY_LIST_RESTART_FEATURES_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_IMPORT_MEMORY_ZIRCON_HANDLE_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_MEMORY_ZIRCON_HANDLE_PROPERTIES_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_MEMORY_GET_ZIRCON_HANDLE_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_ZIRCON_HANDLE_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_SEMAPHORE_GET_ZIRCON_HANDLE_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_BUFFER_COLLECTION_CREATE_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_IMPORT_MEMORY_BUFFER_COLLECTION_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_BUFFER_COLLECTION_IMAGE_CREATE_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_BUFFER_COLLECTION_PROPERTIES_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_BUFFER_CONSTRAINTS_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_BUFFER_COLLECTION_BUFFER_CREATE_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_IMAGE_CONSTRAINTS_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_IMAGE_FORMAT_CONSTRAINTS_INFO_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_SYSMEM_COLOR_SPACE_FUCHSIA),
    	// ENTRY(VK_STRUCTURE_TYPE_BUFFER_COLLECTION_CONSTRAINTS_INFO_FUCHSIA),
    	ENTRY(VK_STRUCTURE_TYPE_SUBPASS_SHADING_PIPELINE_CREATE_INFO_HUAWEI),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBPASS_SHADING_FEATURES_HUAWEI),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBPASS_SHADING_PROPERTIES_HUAWEI),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INVOCATION_MASK_FEATURES_HUAWEI),
    	ENTRY(VK_STRUCTURE_TYPE_MEMORY_GET_REMOTE_ADDRESS_INFO_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_RDMA_FEATURES_NV),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTENDED_DYNAMIC_STATE_2_FEATURES_EXT),
    	// ENTRY(VK_STRUCTURE_TYPE_SCREEN_SURFACE_CREATE_INFO_QNX),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COLOR_WRITE_ENABLE_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PIPELINE_COLOR_WRITE_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_VIEW_MIN_LOD_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_IMAGE_VIEW_MIN_LOD_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTI_DRAW_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTI_DRAW_PROPERTIES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BORDER_COLOR_SWIZZLE_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_SAMPLER_BORDER_COLOR_COMPONENT_MAPPING_CREATE_INFO_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PAGEABLE_DEVICE_LOCAL_MEMORY_FEATURES_EXT),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_OFFSET_FEATURES_QCOM),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_OFFSET_PROPERTIES_QCOM),
    	ENTRY(VK_STRUCTURE_TYPE_SUBPASS_FRAGMENT_DENSITY_MAP_OFFSET_END_INFO_QCOM),
    	ENTRY(VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_LINEAR_COLOR_ATTACHMENT_FEATURES_NV),
#undef ENTRY
	};

	auto it = structSizes.find(type);
    if (it == structSizes.end()) {
        dlg_error("Unknown pNext chain sType '{}'", u64(type));
        return sizeof(VkBaseInStructure);
    }

	return it->second;
}

std::unique_ptr<std::byte[]> copyChain(const void*& pNext) {
	if(!pNext) {
		return {};
	}

	// first march-through: find needed size
	std::size_t size = 0u;
	auto it = pNext;
	while(it) {
		auto src = static_cast<const VkBaseInStructure*>(it);

		auto ssize = structSize(src->sType);
		dlg_assertm(ssize > 0, "Unknown structure type: {}", src->sType);
		size += ssize;

		it = src->pNext;
	}

	auto buf = std::make_unique<std::byte[]>(size);
	auto offset = std::size_t(0u);

	// second-march-through: copy structure
	VkBaseInStructure* last = nullptr;
	it = pNext;
	while(it) {
		auto src = static_cast<const VkBaseInStructure*>(it);
		auto size = structSize(src->sType);
		dlg_assertm(size > 0, "Unknown structure type: {}", src->sType);

		auto dst = reinterpret_cast<VkBaseInStructure*>(buf.get() + offset);
		// TODO: technicallly UB to not construct object via placement new.
		// In practice, this works everywhere since its only C PODs
		std::memcpy(dst, src, size);
		offset += size;

		if(last) {
			last->pNext = dst;
		} else {
			pNext = dst;
		}

		last = dst;
		it = src->pNext;
	}

	dlg_assert(offset == size);
	return buf;
}

void* copyChain(const void*& pNext, std::unique_ptr<std::byte[]>& buf) {
	if(!pNext) {
		return nullptr;
	}

	buf = copyChain(pNext);
	return static_cast<void*>(buf.get());
}

void* copyChainLocal(ThreadMemScope& memScope, const void* pNext) {
	VkBaseInStructure* last = nullptr;
	void* ret = nullptr;
	auto it = static_cast<const VkBaseInStructure*>(pNext);

	while(it) {
		auto size = structSize(it->sType);
		dlg_assertm_or(size > 0, it = it->pNext; continue,
			"Unknown structure type: {}", it->sType);

		auto dstBuf = memScope.alloc<std::byte>(size);
		auto* dst = reinterpret_cast<VkBaseInStructure*>(dstBuf.data());

		// TODO: technicallly UB to not construct object via placement new.
		// In practice, this works everywhere since its only C PODs
		std::memcpy(dst, it, size);
		dst->pNext = nullptr;

		if(!last) {
			dlg_assert(!ret);
			ret = static_cast<void*>(dst);
		} else {
			last->pNext = dst;
		}

		last = dst;
		it = it->pNext;
	}

	return ret;
}

void writeFile(const char* path, span<const std::byte> buffer, bool binary) {
	dlg_assert(path);
	errno = 0;

	auto* f = std::fopen(path, binary ? "wb" : "w");
	if(!f) {
		dlg_error("Could not open '{}' for writing: {}", path, std::strerror(errno));
		return;
	}

	auto ret = std::fwrite(buffer.data(), 1, buffer.size(), f);
	if(ret != buffer.size()) {
		dlg_error("fwrite on '{}' failed: {}", path, std::strerror(errno));
	}

	std::fclose(f);
}

u32 indexSize(VkIndexType type) {
	// NOTE: When extending here, also extend readIndex
	switch(type) {
		case VK_INDEX_TYPE_UINT16: return 2;
		case VK_INDEX_TYPE_UINT32: return 4;
		case VK_INDEX_TYPE_UINT8_EXT: return 1;
		case VK_INDEX_TYPE_MAX_ENUM:
		case VK_INDEX_TYPE_NONE_KHR:
			return 0;
	}

	return 0;
}

u32 readIndex(VkIndexType type, ReadBuf& data) {
	switch(type) {
		case VK_INDEX_TYPE_UINT16: return read<u16>(data);
		case VK_INDEX_TYPE_UINT32: return read<u32>(data);
		case VK_INDEX_TYPE_UINT8_EXT: return read<u8>(data);
		case VK_INDEX_TYPE_MAX_ENUM:
		case VK_INDEX_TYPE_NONE_KHR:
			dlg_error("invalid index type");
			return 0;
	}

	dlg_error("invalid index type");
	return 0;
}


BufferInterval minMaxInterval(span<const VkBufferImageCopy2KHR> copies, u32 texelSize) {
	VkDeviceSize offset = VkDeviceSize(-1);
	VkDeviceSize end = 0u;

	for(auto& copy : copies) {
		auto stride = copy.bufferRowLength ?
			copy.bufferRowLength : copy.imageExtent.width * texelSize;
		auto hstride = copy.bufferImageHeight ?
			copy.bufferImageHeight : copy.imageExtent.height * stride;
		auto size = VkDeviceSize(hstride * copy.imageExtent.depth);
		offset = std::min(copy.bufferOffset, offset);
		end = std::max(copy.bufferOffset + size, end);
	}

	return {offset, end - offset};
}

BufferInterval minMaxInterval(span<const VkBufferCopy2KHR> copies, bool src) {
	VkDeviceSize offset = VkDeviceSize(-1);
	VkDeviceSize end = 0u;

	for(auto& copy : copies) {
		auto off = src ? copy.srcOffset : copy.dstOffset;
		offset = std::min(off, offset);
		end = std::max(off + copy.size, end);
	}

	return {offset, end - offset};
}

VkImageAspectFlags aspects(VkFormat format) {
	VkImageAspectFlags ret {};
	if(FormatHasDepth(format)) {
		ret |= VK_IMAGE_ASPECT_DEPTH_BIT;
	}
	if(FormatHasStencil(format)) {
		ret |= VK_IMAGE_ASPECT_STENCIL_BIT;
	}
	if(FormatIsColor(format)) {
		ret |= VK_IMAGE_ASPECT_COLOR_BIT;
	}

	return ret;
}

VkImageViewType imageViewForImageType(VkImageType type) {
	switch(type) {
		case VK_IMAGE_TYPE_1D:
			return VK_IMAGE_VIEW_TYPE_1D_ARRAY;
		case VK_IMAGE_TYPE_2D:
			return VK_IMAGE_VIEW_TYPE_2D_ARRAY;
		case VK_IMAGE_TYPE_3D:
			return VK_IMAGE_VIEW_TYPE_3D;
		default:
			dlg_error("Unsupported image type");
			return VK_IMAGE_VIEW_TYPE_MAX_ENUM;
	}
}

u32 combineQueueFamilies(span<const u32> queueFams) {
	u32 ret = 0u;
	for(auto qf : queueFams) {
		dlg_assert(qf < 32);
		ret |= (1u << qf);
	}

	return ret;
}

ShaderImageType::Value ShaderImageType::parseType(VkImageType imgType,
		VkFormat format, VkImageAspectFlagBits aspect) {
	// NOTE: relies on ordering of DrawGuiImage::Type enum
	using NumType = FORMAT_NUMERICAL_TYPE;
	auto imageTypeFUI = [](auto numt) {
		if(numt == NumType::SINT) return Value::i1;
		else if(numt == NumType::UINT) return Value::u1;
		else return Value::f1;
	};

	Value baseType;

	if(aspect == VK_IMAGE_ASPECT_COLOR_BIT) {
		if(FormatIsSampledFloat(format)) baseType = Value::f1;
		else if(FormatIsUINT(format)) baseType = Value::u1;
		else if(FormatIsSINT(format)) baseType = Value::i1;
		else {
			dlg_error("unreachable");
			return ShaderImageType::count;
		}
	} else {
		auto numt = NumType::NONE;
		if(aspect == VK_IMAGE_ASPECT_DEPTH_BIT) {
			numt = FormatDepthNumericalType(format);
		} else if(aspect == VK_IMAGE_ASPECT_STENCIL_BIT) {
			numt = FormatStencilNumericalType(format);
		} else {
			dlg_error("unreachable");
			return ShaderImageType::count;
		}

		baseType = imageTypeFUI(numt);
	}

	auto off = 0u;
	switch(imgType) {
		case VK_IMAGE_TYPE_1D: off = 0u; break;
		case VK_IMAGE_TYPE_2D: off = 1u; break;
		case VK_IMAGE_TYPE_3D: off = 2u; break;
		default: dlg_error("unreachable"); return ShaderImageType::count;
	}

	return Value(unsigned(baseType) + off);
}

} // namespace vil
