---
title: Utility Kit
description: 'Utility kit for easier DevTools integrations.'
---

Since v0.3.0, we are now providing a utility kit for easier DevTools integrations, similar to `@nuxt/kit`.

```bash
npm i @nuxt/devtools-kit
```

```ts
import { addCustomTab } from '@nuxt/devtools-kit'
```

We recommend module authors to install `@nuxt/devtools-kit` as a dependency and `@nuxt/devtools` as a dev dependency.

## `@nuxt/devtools-kit`

### `addCustomTab()`

A shorthand for calling the hook `devtools:customTabs`.

```ts
import { addCustomTab } from '@nuxt/devtools-kit'

addCustomTab(() => ({
  // unique identifier
  name: 'my-module',
  // title to display in the tab
  title: 'My Module',
  // any icon from Iconify, or a URL to an image
  icon: 'carbon:apps',
  // iframe view
  view: {
    type: 'iframe',
    src: '/url-to-your-module-view',
  },
}))
```

### `refreshCustomTabs()`

A shorthand for call hook `devtools:customTabs:refresh`. It will refresh all custom tabs.

### `startSubprocess()`

Start a sub process using `execa` and create a terminal tab in DevTools.

```ts
import { startSubprocess } from '@nuxt/devtools-kit'

const subprocess = startSubprocess(
  {
    command: 'code-server',
    args: [
      'serve-local',
      '--accept-server-license-terms',
      '--without-connection-token',
      `--port=${port}`,
    ],
  },
  {
    id: 'devtools:vscode',
    name: 'VS Code Server',
    icon: 'logos-visual-studio-code',
  },
)
```

```ts
subprocess.restart()
subprocess.terminate()
```

### `extendServerRpc()`

Extend the server RPC with your own methods.

```ts
import { extendServerRpc } from '@nuxt/devtools-kit'

const rpc = extendServerRpc('my-module', {
  async myMethod() {
    return 'hello'
  },
})
```

Learn more about [Custom RPC functions](/module/guide#custom-rpc-functions).

## `@nuxt/devtools-kit/iframe-client`

To provide complex interactions for your module integrations, we recommend to host your own view and display it in devtools via iframe.

To get the infomation from the devtools and the client app, you can do this in your client app:

```ts
import { useDevtoolsClient } from '@nuxt/devtools-kit/iframe-client'

export const devtoolsClient = useDevtoolsClient()
```

When the iframe been served with the same origin (CORS limitation), devtools will automatically inject `__NUXT_DEVTOOLS__` to the iframe's window object. You can access it as a ref using `useDevtoolsClient()` utility.

### `useDevtoolsClient()`

It will return a ref of `NuxtDevtoolsIframeClient` object that are intially `null` and will be updated when the connection is ready.

`NuxtDevtoolsIframeClient` contains two properties:

- `host`: APIs to communicate with the main app in browser
- `devtools`: APIs to communicate with the devtools

`host` can be undefined when devtools are accessed standalone or from a different origin.

For example, you can get the router instance from the client app:

```ts
const router = computed(() => devtoolsClient.value?.host?.nuxt.vueApp.config.globalProperties?.$router)
```

### `onDevtoolsClientConnected()`

Similiar to `useDevtoolsClient()` but as a callback style:

```ts
import { onDevtoolsClientConnected } from '@nuxt/devtools-kit/iframe-client'

onDevtoolsClientConnected(async (client) => {
  // client is NuxtDevtoolsIframeClient

  const config = client.devtools.rpc.getServerConfig()
  // ...
})
```

## `@nuxt/devtools-kit/host-client`

When you have iframe for your devtools view, sometimes you need to communicate with the devtools host (the main app in browser) with a runtime plugin. You can use `@nuxt/devtools-kit/host-client` to do that.

### `useDevtoolsHostClient()`

It will return a ref of `NuxtDevtoolsHostClient` object that are intially `null` and will be updated when the host is initialized by NuxtDevtools.

```ts
import { useDevtoolsHostClient } from '@nuxt/devtools-kit/host-client'

export default defineNuxtPlugin({
  name: 'my-module:devtools',
  setup(nuxtApp) {
    const devtoolsHost = useDevtoolsHostClient()

    // ...
  }
})
```

### `onDevtoolsHostClientConnected()`

Similiar to `useDevtoolsHostClient()` but as a callback style:

```ts
import { onDevtoolsHostClientConnected } from '@nuxt/devtools-kit/host-client'

onDevtoolsHostClientConnected(async (host) => {

})
```
