% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MeasureSurvSchmid.R
\name{mlr_measures_surv.schmid}
\alias{mlr_measures_surv.schmid}
\alias{MeasureSurvSchmid}
\title{Integrated Schmid Score Survival Measure}
\description{
Calculates the \strong{Integrated Schmid Score} (ISS), aka integrated absolute loss.
}
\details{
This measure has two dimensions: (test set) observations and time points.
For a specific individual \eqn{i} from the test set, with observed survival
outcome \eqn{(t_i, \delta_i)} (time and censoring indicator) and predicted
survival function \eqn{S_i(t)}, the \emph{observation-wise} estimator of the loss,
integrated across the time dimension up to the time cutoff \eqn{\tau^*}, is:

\deqn{L_{ISS}(S_i, t_i, \delta_i) = \int^{\tau^*}_0  \frac{S_i(\tau) \text{I}(t_i \leq \tau, \delta=1)}{G(t_i)} + \frac{(1-S_i(\tau)) \text{I}(t_i > \tau)}{G(\tau)} \ d\tau}

where \eqn{G} is the Kaplan-Meier estimate of the censoring distribution.

The implementation uses the trapezoidal rule to approximate the integral over
time and the integral is normalized by the range of available evaluation times
(\eqn{\tau_{\text{max}} - \tau_{\text{min}}}).

To get a single score across all \eqn{N} observations of the test set, we
return the average of the time-integrated observation-wise scores:
\deqn{\sum_{i=1}^N L(S_i, t_i, \delta_i) / N}
}
\section{Dictionary}{

This \link[mlr3:Measure]{Measure} can be instantiated via the \link[mlr3misc:Dictionary]{dictionary}
\link[mlr3:mlr_measures]{mlr_measures} or with the associated sugar function \link[mlr3:mlr_sugar]{msr()}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{MeasureSurvSchmid$new()
mlr_measures$get("surv.schmid")
msr("surv.schmid")
}\if{html}{\out{</div>}}
}

\section{Parameters}{
\tabular{lllll}{
   Id \tab Type \tab Default \tab Levels \tab Range \cr
   integrated \tab logical \tab TRUE \tab TRUE, FALSE \tab - \cr
   times \tab untyped \tab - \tab  \tab - \cr
   t_max \tab numeric \tab - \tab  \tab \eqn{[0, \infty)}{[0, Inf)} \cr
   p_max \tab numeric \tab - \tab  \tab \eqn{[0, 1]}{[0, 1]} \cr
   eps \tab numeric \tab 0.001 \tab  \tab \eqn{[0, 1]}{[0, 1]} \cr
   ERV \tab logical \tab FALSE \tab TRUE, FALSE \tab - \cr
}
}

\section{Meta Information}{

\itemize{
\item Type: \code{"surv"}
\item Range: \eqn{[0, \infty)}{[0, Inf)}
\item Minimize: \code{TRUE}
\item Required prediction: \code{distr}
}
}

\section{Parameter details}{

\itemize{
\item \code{integrated} (\code{logical(1)})\cr
If \code{TRUE} (default), returns the integrated score (eg across
time points); otherwise, not integrated (eg at a single time point).
}


\itemize{
\item \code{times} (\code{numeric()})\cr
If \code{integrated == TRUE} then a vector of time-points over which to integrate the score.
If \code{integrated == FALSE} then a single time point at which to return the score.
}


\itemize{
\item \code{t_max} (\code{numeric(1)})\cr
Cutoff time \eqn{\tau^*} (i.e. time horizon) to evaluate the measure up to
(truncate \eqn{S(t)}).
Mutually exclusive with \code{p_max} or \code{times}.
It's recommended to set \code{t_max} to avoid division by \code{eps}, see "Time Cutoff Details" section.
If \code{t_max} is not specified, an \code{Inf} time horizon is assumed.
}


\itemize{
\item \code{p_max} (\code{numeric(1)})\cr
The proportion of censoring to integrate up to in the given dataset.
Mutually exclusive with \code{times} or \code{t_max}.
}


\itemize{
\item \code{eps} (\code{numeric(1)})\cr
Very small number to substitute near-zero values in order to prevent errors
in e.g. log(0) and/or division-by-zero calculations.
Default value is 0.001.
}


\itemize{
\item \code{ERV} (\code{logical(1)})\cr
If \code{TRUE} then the Explained Residual Variation method is applied, which
means the score is standardized against a Kaplan-Meier baseline.
Default is \code{FALSE}.
}
}

\section{Properness}{


ISS is not a proper scoring rule, see Sonabend et al. (2024) for more details.
The assumptions for consistent estimation of the loss are that the censoring
distribution \eqn{G(t)} is independent of the survival distribution and \eqn{G(t)}
is fit on a sufficiently large dataset.
}

\section{Time points used for evaluation}{


If the \code{times} argument is not specified (\code{NULL}), then the sorted unique
time points from the \strong{test set} are used for evaluation of the
time-integrated score.
This was a design decision due to the fact that different predicted survival
distributions \eqn{S(t)} usually have a \strong{discretized time domain} which may
differ, i.e. in the case the survival predictions come from different survival
learners.
Essentially, using the same set of time points for the calculation of the score
minimizes the bias that would come from using different time points.
We note that we perform \strong{constant interpolation} of \eqn{S(t)} for time points
that fall outside its discretized time domain.

Naturally, if the \code{times} argument is specified, then exactly these time
points are used for evaluation.
A warning is given to the user in case some of the specified \code{times} fall outside
of the time point range of the test set.
The assumption here is that if the test set is large enough, it should have a
time domain/range similar to the one from the train set, and therefore time
points outside that domain might lead to unwanted extrapolation of \eqn{S(t)}.
}

\section{Data used for Estimating Censoring Distribution}{


If \code{task} and \code{train_set} are passed to \verb{$score} then \eqn{G(t)} is fit using
\strong{all observations} from the train set, otherwise the test set is used.
Using the train set is likely to reduce any bias caused by calculating parts of the
measure on the test data it is evaluating.
Also usually it means that more data is used for fitting the censoring
distribution \eqn{G(t)} via the Kaplan-Meier.
The training data is automatically used in scoring resamplings.
}

\section{Time Cutoff Details}{


If \code{t_max} or \code{p_max} is given, then the predicted survival function \eqn{S(t)} is
truncated at the time cutoff for all observations. This helps mitigate
\strong{inflation of the score} which can occur when an observation is censored
at the last observed time. In such cases, \eqn{G(t) = 0}, triggering the use
of a small constant \code{eps} instead, see Kvamme et al. (2023).
Not using a \code{t_max} can lead to misleading evaluation, violations of properness
and poor optimization outcomes when using this score for model tuning, see
Sonabend et al. (2024).
}

\section{Implementation differences}{


If comparing the integrated Graf score to other packages, e.g. \CRANpkg{pec},
results may be very slightly different as this package uses \code{survfit} to estimate
the censoring distribution, in line with the Graf 1999 paper; whereas some
other packages use \code{prodlim} with \code{reverse = TRUE} (meaning Kaplan-Meier is
not used).
}

\examples{
library(mlr3)

# Define a survival Task
task = tsk("lung")

# Create train and test set
part = partition(task)

# Train Cox learner on the train set
cox = lrn("surv.coxph")
cox$train(task, row_ids = part$train)

# Make predictions for the test set
p = cox$predict(task, row_ids = part$test)

# ISS, G(t) calculated using the test set
p$score(msr("surv.schmid"))

# ISS, G(t) calculated using the train set (always recommended)
p$score(msr("surv.schmid"), task = task, train_set = part$train)

# ISS, ERV score (comparing with KM baseline)
p$score(msr("surv.schmid", ERV = TRUE), task = task, train_set = part$train)

# ISS at specific time point
p$score(msr("surv.schmid", times = 365), task = task, train_set = part$train)

# ISS at multiple time points (integrated)
p$score(msr("surv.schmid", times = c(125, 365, 450), integrated = TRUE),
        task = task, train_set = part$train)

# ISS, use time cutoff
p$score(msr("surv.schmid", t_max = 700), task = task, train_set = part$train)

# ISS, use time cutoff corresponding to specific proportion of censoring on the test set
p$score(msr("surv.schmid", p_max = 0.8), task = task, train_set = part$train)

}
\references{
Schemper, Michael, Henderson, Robin (2000).
\dQuote{Predictive Accuracy and Explained Variation in Cox Regression.}
\emph{Biometrics}, \bold{56}, 249--255.
\doi{10.1002/sim.1486}.

Schmid, Matthias, Hielscher, Thomas, Augustin, Thomas, Gefeller, Olaf (2011).
\dQuote{A Robust Alternative to the Schemper-Henderson Estimator of Prediction Error.}
\emph{Biometrics}, \bold{67}(2), 524--535.
\doi{10.1111/j.1541-0420.2010.01459.x}.

Sonabend, Raphael, Zobolas, John, Kopper, Philipp, Burk, Lukas, Bender, Andreas (2024).
\dQuote{Examining properness in the external validation of survival models with squared and logarithmic losses.}
\url{https://arxiv.org/abs/2212.05260v3}.

Kvamme, Havard, Borgan, Ornulf (2023).
\dQuote{The Brier Score under Administrative Censoring: Problems and a Solution.}
\emph{Journal of Machine Learning Research}, \bold{24}(2), 1--26.
ISSN 1533-7928, \url{http://jmlr.org/papers/v24/19-1030.html}.
}
\seealso{
Other survival measures: 
\code{\link{mlr_measures_surv.calib_alpha}},
\code{\link{mlr_measures_surv.calib_beta}},
\code{\link{mlr_measures_surv.calib_index}},
\code{\link{mlr_measures_surv.chambless_auc}},
\code{\link{mlr_measures_surv.cindex}},
\code{\link{mlr_measures_surv.dcalib}},
\code{\link{mlr_measures_surv.graf}},
\code{\link{mlr_measures_surv.hung_auc}},
\code{\link{mlr_measures_surv.intlogloss}},
\code{\link{mlr_measures_surv.logloss}},
\code{\link{mlr_measures_surv.mae}},
\code{\link{mlr_measures_surv.mse}},
\code{\link{mlr_measures_surv.nagelk_r2}},
\code{\link{mlr_measures_surv.oquigley_r2}},
\code{\link{mlr_measures_surv.rcll}},
\code{\link{mlr_measures_surv.rmse}},
\code{\link{mlr_measures_surv.song_auc}},
\code{\link{mlr_measures_surv.song_tnr}},
\code{\link{mlr_measures_surv.song_tpr}},
\code{\link{mlr_measures_surv.uno_auc}},
\code{\link{mlr_measures_surv.uno_tnr}},
\code{\link{mlr_measures_surv.uno_tpr}},
\code{\link{mlr_measures_surv.xu_r2}}

Other Probabilistic survival measures: 
\code{\link{mlr_measures_surv.graf}},
\code{\link{mlr_measures_surv.intlogloss}},
\code{\link{mlr_measures_surv.logloss}},
\code{\link{mlr_measures_surv.rcll}}

Other distr survival measures: 
\code{\link{mlr_measures_surv.calib_alpha}},
\code{\link{mlr_measures_surv.calib_index}},
\code{\link{mlr_measures_surv.dcalib}},
\code{\link{mlr_measures_surv.graf}},
\code{\link{mlr_measures_surv.intlogloss}},
\code{\link{mlr_measures_surv.logloss}},
\code{\link{mlr_measures_surv.rcll}}
}
\concept{Probabilistic survival measures}
\concept{distr survival measures}
\concept{survival measures}
\section{Super classes}{
\code{\link[mlr3:Measure]{mlr3::Measure}} -> \code{\link[mlr3proba:MeasureSurv]{mlr3proba::MeasureSurv}} -> \code{MeasureSurvSchmid}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-MeasureSurvSchmid-new}{\code{MeasureSurvSchmid$new()}}
\item \href{#method-MeasureSurvSchmid-clone}{\code{MeasureSurvSchmid$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="aggregate"><a href='../../mlr3/html/Measure.html#method-Measure-aggregate'><code>mlr3::Measure$aggregate()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="format"><a href='../../mlr3/html/Measure.html#method-Measure-format'><code>mlr3::Measure$format()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="help"><a href='../../mlr3/html/Measure.html#method-Measure-help'><code>mlr3::Measure$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="print"><a href='../../mlr3/html/Measure.html#method-Measure-print'><code>mlr3::Measure$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="score"><a href='../../mlr3/html/Measure.html#method-Measure-score'><code>mlr3::Measure$score()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MeasureSurvSchmid-new"></a>}}
\if{latex}{\out{\hypertarget{method-MeasureSurvSchmid-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MeasureSurvSchmid$new(ERV = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{ERV}}{(\code{logical(1)})\cr
Standardize measure against a Kaplan-Meier baseline
(Explained Residual Variation)}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MeasureSurvSchmid-clone"></a>}}
\if{latex}{\out{\hypertarget{method-MeasureSurvSchmid-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MeasureSurvSchmid$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
