% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MeasureSurvICI.R
\name{mlr_measures_surv.calib_index}
\alias{mlr_measures_surv.calib_index}
\alias{MeasureSurvICI}
\title{Integrated Calibration Index Survival Measure}
\description{
Calculates the Integrated Calibration Index (ICI), which evaluates
\strong{point-calibration} (i.e. at a specific time point), see Austin et al. (2020).
}
\details{
Each individual \eqn{i} from the test set, has an observed survival outcome
\eqn{(t_i, \delta_i)} (time and censoring indicator) and predicted survival
function \eqn{S_i(t)}.
The predicted probability of an event occurring before a specific time point
\eqn{t_0}, is defined as \eqn{\hat{P_i}(t_0) = F_i(t_0) = 1 - S_i(t_0)}.

Using hazard regression (via the \CRANpkg{polspline} R package), a \emph{smoothed}
calibration curve is estimated by fitting the following model:
\deqn{log(h(t)) = g(log(− log(1 − \hat{P}_{t_0})), t)}

Note that we substitute probabilities \eqn{\hat{P}_{t_0} = 0} with a small
\eqn{\epsilon} number to avoid arithmetic issues (\eqn{log(0)}). Same with
\eqn{\hat{P}_{t_0} = 1}, we use \eqn{1 - \epsilon}.
From this model, the \emph{smoothed} probability of occurrence at \eqn{t_0} for
observation \eqn{i} is obtained as \eqn{\hat{P}_i^c(t_0)}.

The \strong{Integrated Calibration Index} is then computed across the \eqn{N}
test set observations as:
\deqn{ICI = \frac{1}{N} \sum_{i=1}^N | \hat{P}_i^c(t_0) - \hat{P}_i(t_0) |}

Therefore, a perfect calibration (smoothed probabilities match predicted
probabilities for all observations) yields \eqn{ICI = 0}, while the worst
possible score is \eqn{ICI = 1}.
}
\section{Dictionary}{

This \link[mlr3:Measure]{Measure} can be instantiated via the \link[mlr3misc:Dictionary]{dictionary}
\link[mlr3:mlr_measures]{mlr_measures} or with the associated sugar function \link[mlr3:mlr_sugar]{msr()}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{MeasureSurvICI$new()
mlr_measures$get("surv.calib_index")
msr("surv.calib_index")
}\if{html}{\out{</div>}}
}

\section{Parameters}{
\tabular{lllll}{
   Id \tab Type \tab Default \tab Levels \tab Range \cr
   time \tab numeric \tab - \tab  \tab \eqn{[0, \infty)}{[0, Inf)} \cr
   eps \tab numeric \tab 1e-04 \tab  \tab \eqn{[0, 1]}{[0, 1]} \cr
   method \tab character \tab ICI \tab ICI, E50, E90, Emax \tab - \cr
   na.rm \tab logical \tab TRUE \tab TRUE, FALSE \tab - \cr
}
}

\section{Meta Information}{

\itemize{
\item Type: \code{"surv"}
\item Range: \eqn{[0, 1]}{[0, 1]}
\item Minimize: \code{TRUE}
\item Required prediction: \code{distr}
}
}

\section{Parameter details}{

\itemize{
\item \code{eps} (\code{numeric(1)})\cr
Very small number to substitute near-zero values in order to prevent errors
in e.g. log(0) and/or division-by-zero calculations.
Default value is 1e-04.
}


\itemize{
\item \code{time} (\code{numeric(1)})\cr
The specific time point \eqn{t_0} at which calibration is evaluated.
If \code{NULL}, the median observed time from the test set is used.
\item \code{method} (\code{character(1)})\cr
Specifies the summary statistic used to calculate the final calibration score.
\itemize{
\item \code{"ICI"} (default): Uses the mean of absolute differences \eqn{| \hat{P}_i^c(t_0) - \hat{P}_i(t_0) |} across all observations.
\item \code{"E50"}: Uses the median of absolute differences instead of the mean.
\item \code{"E90"}: Uses the 90th percentile of absolute differences, emphasizing higher deviations.
\item \code{"Emax"}: Uses the maximum absolute difference, capturing the largest discrepancy between predicted and smoothed probabilities.
}
\item \code{na.rm} (\code{logical(1)})\cr
If \code{TRUE} (default) then removes any NAs/NaNs in the smoothed probabilities
\eqn{\hat{P}_i^c(t_0)} that may arise. A warning is issued nonetheless in such
cases.
}
}

\examples{
library(mlr3)

# Define a survival Task
task = tsk("lung")

# Create train and test set
part = partition(task)

# Train Cox learner on the train set
cox = lrn("surv.coxph")
cox$train(task, row_ids = part$train)

# Make predictions for the test set
p = cox$predict(task, row_ids = part$test)

# ICI at median test set time
p$score(msr("surv.calib_index"))

# ICI at specific time point
p$score(msr("surv.calib_index", time = 365))

# E50 at specific time point
p$score(msr("surv.calib_index", method = "E50", time = 365))

}
\references{
Austin, C. P, Harrell, E. F, van Klaveren, David (2020).
\dQuote{Graphical calibration curves and the integrated calibration index (ICI) for survival models.}
\emph{Statistics in Medicine}, \bold{39}(21), 2714.
ISSN 10970258, \doi{10.1002/SIM.8570}, \url{https://pmc.ncbi.nlm.nih.gov/articles/PMC7497089/}.
}
\seealso{
Other survival measures: 
\code{\link{mlr_measures_surv.calib_alpha}},
\code{\link{mlr_measures_surv.calib_beta}},
\code{\link{mlr_measures_surv.chambless_auc}},
\code{\link{mlr_measures_surv.cindex}},
\code{\link{mlr_measures_surv.dcalib}},
\code{\link{mlr_measures_surv.graf}},
\code{\link{mlr_measures_surv.hung_auc}},
\code{\link{mlr_measures_surv.intlogloss}},
\code{\link{mlr_measures_surv.logloss}},
\code{\link{mlr_measures_surv.mae}},
\code{\link{mlr_measures_surv.mse}},
\code{\link{mlr_measures_surv.nagelk_r2}},
\code{\link{mlr_measures_surv.oquigley_r2}},
\code{\link{mlr_measures_surv.rcll}},
\code{\link{mlr_measures_surv.rmse}},
\code{\link{mlr_measures_surv.schmid}},
\code{\link{mlr_measures_surv.song_auc}},
\code{\link{mlr_measures_surv.song_tnr}},
\code{\link{mlr_measures_surv.song_tpr}},
\code{\link{mlr_measures_surv.uno_auc}},
\code{\link{mlr_measures_surv.uno_tnr}},
\code{\link{mlr_measures_surv.uno_tpr}},
\code{\link{mlr_measures_surv.xu_r2}}

Other calibration survival measures: 
\code{\link{mlr_measures_surv.calib_alpha}},
\code{\link{mlr_measures_surv.calib_beta}},
\code{\link{mlr_measures_surv.dcalib}}

Other distr survival measures: 
\code{\link{mlr_measures_surv.calib_alpha}},
\code{\link{mlr_measures_surv.dcalib}},
\code{\link{mlr_measures_surv.graf}},
\code{\link{mlr_measures_surv.intlogloss}},
\code{\link{mlr_measures_surv.logloss}},
\code{\link{mlr_measures_surv.rcll}},
\code{\link{mlr_measures_surv.schmid}}
}
\concept{calibration survival measures}
\concept{distr survival measures}
\concept{survival measures}
\section{Super classes}{
\code{\link[mlr3:Measure]{mlr3::Measure}} -> \code{\link[mlr3proba:MeasureSurv]{mlr3proba::MeasureSurv}} -> \code{MeasureSurvICI}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-MeasureSurvICI-new}{\code{MeasureSurvICI$new()}}
\item \href{#method-MeasureSurvICI-clone}{\code{MeasureSurvICI$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="aggregate"><a href='../../mlr3/html/Measure.html#method-Measure-aggregate'><code>mlr3::Measure$aggregate()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="format"><a href='../../mlr3/html/Measure.html#method-Measure-format'><code>mlr3::Measure$format()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="help"><a href='../../mlr3/html/Measure.html#method-Measure-help'><code>mlr3::Measure$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="print"><a href='../../mlr3/html/Measure.html#method-Measure-print'><code>mlr3::Measure$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Measure" data-id="score"><a href='../../mlr3/html/Measure.html#method-Measure-score'><code>mlr3::Measure$score()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MeasureSurvICI-new"></a>}}
\if{latex}{\out{\hypertarget{method-MeasureSurvICI-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MeasureSurvICI$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MeasureSurvICI-clone"></a>}}
\if{latex}{\out{\hypertarget{method-MeasureSurvICI-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MeasureSurvICI$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
