% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pipelines.R
\name{mlr_graphs_survtoregr_pem}
\alias{mlr_graphs_survtoregr_pem}
\alias{pipeline_survtoregr_pem}
\title{Survival to Poisson Regression Reduction Pipeline}
\usage{
pipeline_survtoregr_pem(
  learner,
  cut = NULL,
  max_time = NULL,
  graph_learner = FALSE
)
}
\arguments{
\item{learner}{\link[mlr3:LearnerRegr]{LearnerRegr}\cr
Regression learner to fit the transformed \link[mlr3:TaskRegr]{TaskRegr}.
\code{learner} must be able to handle \code{offset} and support optimization of a poisson likelihood.}

\item{cut}{\code{numeric()}\cr
Split points, used to partition the data into intervals.
If unspecified, all unique event times will be used.
If \code{cut} is a single integer, it will be interpreted as the number of equidistant
intervals from 0 until the maximum event time.}

\item{max_time}{\code{numeric(1)}\cr
If cut is unspecified, this will be the last possible event time.
All event times after max_time will be administratively censored at max_time.}

\item{graph_learner}{\code{logical(1)}\cr
If \code{TRUE} returns wraps the \link[mlr3pipelines:Graph]{Graph} as a
\link[mlr3pipelines:mlr_learners_graph]{GraphLearner} otherwise (default) returns as a \code{Graph}.}
}
\value{
\link[mlr3pipelines:Graph]{mlr3pipelines::Graph} or \link[mlr3pipelines:mlr_learners_graph]{mlr3pipelines::GraphLearner}
}
\description{
Wrapper around multiple \link[mlr3pipelines:PipeOp]{PipeOp}s to help in creation
of complex survival reduction methods.
}
\details{
A brief mathematical summary of PEMs (see referenced article for more detail):
\enumerate{
\item{\strong{PED Transformation: }
Survival data is converted into piece-wise exponential data (PED) format.
Key elements are: Continuous time is divided into \eqn{j = 1, \ldots, J} intervals for each subject, \eqn{i = 1, \ldots, n}.
A status variable in each entry indicates whether an event or censoring occurred during that interval. For any subject, data entries are
created only up until the interval including the event time. An offset column is introduced and represents the logarithm of the time a subject spent in any given interval.
For more details, see \code{\link[pammtools:as_ped]{pammtools::as_ped()}}.
}
\item{\strong{Hazard Estimation with PEM: }
The PED transformation combined with the working assumption \deqn{\delta_{ij} \stackrel{\text{iid}}{\sim} Poisson \left( \mu_{ij} = \lambda_{ij} t_{ij} \right),}
where \eqn{\delta_{ij}} denotes the event or censoring indicator, allows framing the problem of piecewise constant hazard estimation as a poisson regression with offset.
Specifically, we want to estimate \deqn{\lambda(t \mid \mathbf{x}_i) := exp(g(x_{i},t_{j})), \quad \forall t \in [t_{j-1}, t_{j}), \quad i = 1, \dots, n.}
\eqn{g(x_{i},t_{j})} is a general function of features \eqn{x} and \eqn{t}, i.e. a learner, and may include non-linearity and complex feature interactions.
Two important prerequisites of the learner are its capacity to model a poisson likelihood and accommodate the offset.
}
\item{\strong{From Piecewise Hazards to Survival Probabilities: }
Lastly, the computed hazards are back transformed to survival probabilities via the following identity
\deqn{S(t | \mathbf{x}) = \exp \left( - \int_{0}^{t} \lambda(s | \mathbf{x}) \, ds \right) = \exp \left( - \sum_{j = 1}^{J} \lambda(j | \mathbf{x}) d_j\,  \right),}
where \eqn{d_j} specifies the duration of interval \eqn{j}.
}
}
The previous considerations are reflected in the pipeline which consists of the following steps:
\enumerate{
\item \link{PipeOpTaskSurvRegrPEM} Converts \link{TaskSurv} to a \link[mlr3:TaskRegr]{TaskRegr}.
\item A \link{LearnerRegr} is fit and predicted on the new \code{TaskRegr}.
\item \link{PipeOpPredRegrSurvPEM} transforms the resulting \link[mlr3:PredictionRegr]{PredictionRegr}
to \link{PredictionSurv}.
}
}
\examples{
\dontshow{if (mlr3misc::require_namespaces(c("mlr3pipelines", "mlr3learners"), quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\dontrun{
  library(mlr3)
  library(mlr3learners)
  library(mlr3pipelines)

  task = tsk("lung")
  part = partition(task)

  # typically model formula and features types are extracted from the task
  learner = lrn("regr.gam", family = "poisson")
  grlrn = ppl(
   "survtoregr_pem",
    learner = learner,
    graph_learner = TRUE
  )
  grlrn$train(task, row_ids = part$train)
  grlrn$predict(task, row_ids = part$test)

  # In some instances special formulas can be specified in the learner
  learner = lrn("regr.gam", family = "poisson", formula = pem_status ~ s(tend) + s(age) + meal.cal)
  grlrn = ppl(
   "survtoregr_pem",
    learner = learner,
    graph_learner = TRUE
  )
  grlrn$train(task, row_ids = part$train)
  grlrn$predict(task, row_ids = part$test)

  # if necessary encode data before passing to learner with e.g. po("encode"), po("modelmatrix"), etc.
  # with po("modelmatrix") feature types and formula can be adjusted at the same time
  cut = round(seq(0, max(task$data()$time), length.out = 20))
  learner = as_learner(
    po("modelmatrix", formula = ~ as.factor(tend) + .) \%>>\%
    lrn("regr.glmnet", family = "poisson", lambda = 0)
  )
  grlrn = ppl(
    "survtoregr_pem",
    learner = learner,
    cut = cut,
    graph_learner = TRUE
  )
  grlrn$train(task, row_ids = part$train)
  grlrn$predict(task, row_ids = part$test)

  # xgboost regression learner
  learner = as_learner(
    po("modelmatrix", formula = ~ .) \%>>\%
    lrn("regr.xgboost", objective = "count:poisson", nrounds = 100, eta = 0.1)
  )

  grlrn = ppl(
    "survtoregr_pem",
    learner = learner,
    graph_learner = TRUE
  )
  grlrn$train(task, row_ids = part$train)
  grlrn$predict(task, row_ids = part$test)
}
\dontshow{\}) # examplesIf}
}
\references{
Bender, Andreas, Groll, Andreas, Scheipl, Fabian (2018).
\dQuote{A generalized additive model approach to time-to-event analysis.}
\emph{Statistical Modelling}, \bold{18}(3-4), 299--321.
\url{https://doi.org/10.1177/1471082X17748083}.
}
\seealso{
Other pipelines: 
\code{\link{mlr_graphs_crankcompositor}},
\code{\link{mlr_graphs_distrcompositor}},
\code{\link{mlr_graphs_probregr}},
\code{\link{mlr_graphs_responsecompositor}},
\code{\link{mlr_graphs_survaverager}},
\code{\link{mlr_graphs_survbagging}},
\code{\link{mlr_graphs_survtoclassif_IPCW}},
\code{\link{mlr_graphs_survtoclassif_disctime}}
}
\concept{pipelines}
