<?php

namespace wcf\system\package\plugin;

use wcf\system\application\ApplicationHandler;
use wcf\system\exception\SystemException;
use wcf\system\package\PackageArchive;
use wcf\system\package\PackageInstallationSQLParser;
use wcf\system\WCF;

/**
 * Executes the delivered sql file.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 */
class SQLPackageInstallationPlugin extends AbstractPackageInstallationPlugin
{
    /**
     * @inheritDoc
     */
    public $tableName = 'package_installation_sql_log';

    /**
     * @inheritDoc
     */
    public function install()
    {
        parent::install();

        // extract sql file from archive
        if ($queries = $this->getSQL($this->instruction['value'])) {
            // replace app1_ with app{WCF_N}_ in the table names for all applications
            $queries = ApplicationHandler::insertRealDatabaseTableNames($queries, true);

            // check queries
            $parser = new PackageInstallationSQLParser(
                $queries,
                $this->installation->getPackage(),
                $this->installation->getAction()
            );
            $conflicts = $parser->test();
            if ($conflicts !== [] && isset($conflicts['CREATE TABLE'])) {
                $unknownCreateTable = $conflicts['CREATE TABLE'];

                $errorMessage = "Can't overwrite unknown table";
                if (\count($unknownCreateTable) > 1) {
                    $errorMessage .= "s";
                }
                $errorMessage .= " '" . \implode("', '", $unknownCreateTable) . "'";

                throw new SystemException($errorMessage);
            }

            // execute queries
            $parser->execute();

            // log changes
            $parser->log();
        }
    }

    /**
     * @inheritDoc
     */
    public function uninstall()
    {
        // get logged sql tables/columns
        $sql = "SELECT      wcf1_package_installation_sql_log.*,
                            CASE WHEN sqlIndex <> '' THEN 1 ELSE 0 END AS isIndex,
                            CASE WHEN sqlColumn <> '' THEN 1 ELSE 0 END AS isColumn,
                            CASE WHEN SUBSTRING(sqlIndex, -3) = '_fk' THEN 1 ELSE 0 END AS isForeignKey
                FROM        wcf1_package_installation_sql_log
                WHERE       packageID = ?
                ORDER BY    isIndex DESC,
                            isForeignKey DESC,
                            sqlIndex,
                            isColumn DESC,
                            sqlColumn";
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute([$this->installation->getPackageID()]);
        $entries = $statement->fetchAll(\PDO::FETCH_ASSOC);

        // get all tablenames from database
        $existingTableNames = WCF::getDB()->getEditor()->getTableNames();

        // delete or alter tables
        foreach ($entries as $entry) {
            // don't alter table if it should be dropped
            if (!empty($entry['sqlColumn'])) {
                $isDropped = false;
                foreach ($entries as $entry2) {
                    if (
                        $entry['sqlTable'] == $entry2['sqlTable']
                        && empty($entry2['sqlColumn'])
                        && empty($entry2['sqlIndex'])
                    ) {
                        $isDropped = true;
                    }
                }
                if ($isDropped) {
                    continue;
                }
            }
            // drop table
            if (!empty($entry['sqlTable']) && empty($entry['sqlColumn']) && empty($entry['sqlIndex'])) {
                WCF::getDB()->getEditor()->dropTable($entry['sqlTable']);
            } // drop column
            elseif (\in_array($entry['sqlTable'], $existingTableNames) && !empty($entry['sqlColumn'])) {
                WCF::getDB()->getEditor()->dropColumn($entry['sqlTable'], $entry['sqlColumn']);
            } // drop index
            elseif (\in_array($entry['sqlTable'], $existingTableNames) && !empty($entry['sqlIndex'])) {
                if (\substr($entry['sqlIndex'], -3) == '_fk') {
                    WCF::getDB()->getEditor()->dropForeignKey($entry['sqlTable'], $entry['sqlIndex']);
                } else {
                    WCF::getDB()->getEditor()->dropIndex($entry['sqlTable'], $entry['sqlIndex']);
                }
            }
        }
        // delete from log table
        parent::uninstall();
    }

    /**
     * Extracts and returns the sql file.
     * If the specified sql file was not found, an error message is thrown.
     *
     * @param string $filename
     * @return  string
     * @throws  SystemException
     */
    protected function getSQL($filename)
    {
        // search sql files in package archive
        if (($fileindex = $this->installation->getArchive()->getTar()->getIndexByFilename($filename)) === false) {
            throw new SystemException("SQL file '" . $filename . "' not found.");
        }

        // extract sql file to string
        return $this->installation->getArchive()->getTar()->extractToString($fileindex);
    }

    /**
     * @inheritDoc
     */
    public static function getDefaultFilename()
    {
        return 'install.sql';
    }

    /**
     * @inheritDoc
     */
    public static function isValid(PackageArchive $packageArchive, $instruction)
    {
        if (!$instruction) {
            $instruction = static::getDefaultFilename();
        }

        if (\preg_match('~\.sql$~', $instruction)) {
            // check if file actually exists
            try {
                if ($packageArchive->getTar()->getIndexByFilename($instruction) === false) {
                    return false;
                }
            } catch (SystemException $e) {
                return false;
            }

            return true;
        }

        return false;
    }
}
