<?php

namespace wcf\system\gridView\admin;

use wcf\acp\form\LabelEditForm;
use wcf\data\DatabaseObject;
use wcf\data\label\group\LabelGroupList;
use wcf\data\label\group\ViewableLabelGroup;
use wcf\data\label\I18nLabelList;
use wcf\data\label\Label;
use wcf\event\gridView\admin\LabelGridViewInitialized;
use wcf\system\cache\builder\LabelCacheBuilder;
use wcf\system\gridView\AbstractGridView;
use wcf\system\gridView\GridViewColumn;
use wcf\system\gridView\GridViewRowLink;
use wcf\system\gridView\renderer\AbstractColumnRenderer;
use wcf\system\gridView\renderer\NumberColumnRenderer;
use wcf\system\gridView\renderer\ObjectIdColumnRenderer;
use wcf\system\interaction\admin\LabelInteractions;
use wcf\system\interaction\bulk\admin\LabelBulkInteractions;
use wcf\system\interaction\Divider;
use wcf\system\interaction\EditInteraction;
use wcf\system\view\filter\I18nTextFilter;
use wcf\system\view\filter\IntegerFilter;
use wcf\system\view\filter\SelectFilter;
use wcf\system\WCF;
use wcf\util\StringUtil;

/**
 * Grid view for the list of labels.
 *
 * @author Olaf Braun
 * @copyright 2001-2025 WoltLab GmbH
 * @license GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @since 6.2
 *
 * @extends AbstractGridView<Label, I18nLabelList>
 */
final class LabelGridView extends AbstractGridView
{
    public function __construct()
    {
        $this->addColumns([
            GridViewColumn::for("labelID")
                ->label("wcf.global.objectID")
                ->renderer(new ObjectIdColumnRenderer())
                ->sortable(),
            GridViewColumn::for("label")
                ->label("wcf.acp.label.label")
                ->titleColumn()
                ->filter(I18nTextFilter::class)
                ->renderer(
                    new class extends AbstractColumnRenderer {
                        #[\Override]
                        public function render(mixed $value, DatabaseObject $row): string
                        {
                            \assert($row instanceof Label);
                            $badgeClassName = StringUtil::encodeHTML($row->getClassNames());
                            $label = StringUtil::encodeHTML($row->getTitle());

                            return <<<HTML
                                <span class="badge label {$badgeClassName}">{$label}</span>
                            HTML;
                        }
                    }
                )
                ->sortable(sortByDatabaseColumn: "labelI18n"),
            GridViewColumn::for("groupID")
                ->label("wcf.acp.label.group")
                ->renderer(
                    new class extends AbstractColumnRenderer {
                        #[\Override]
                        public function render(mixed $value, DatabaseObject $row): string
                        {
                            \assert($row instanceof Label);

                            $groups = LabelCacheBuilder::getInstance()->getData(arrayIndex: "groups");
                            $group = $groups[$row->groupID];
                            \assert($group instanceof ViewableLabelGroup);

                            return StringUtil::encodeHTML($group->getExtendedTitle());
                        }
                    }
                )
                ->filter(
                    new SelectFilter(
                        $this->getAvailableLabelGroups(),
                        'groupID',
                        'wcf.acp.label.group',
                        labelLanguageItems: false
                    )
                )
                ->sortable(),
            GridViewColumn::for("showOrder")
                ->label("wcf.global.showOrder")
                ->renderer(new NumberColumnRenderer())
                ->filter(IntegerFilter::class)
                ->sortable(),
        ]);

        $provider = new LabelInteractions();
        $provider->addInteractions([
            new Divider(),
            new EditInteraction(LabelEditForm::class),
        ]);
        $this->setInteractionProvider($provider);
        $this->setBulkInteractionProvider(new LabelBulkInteractions());

        $this->setDefaultSortField("label");
        $this->addRowLink(new GridViewRowLink(LabelEditForm::class));
    }

    #[\Override]
    public function isAccessible(): bool
    {
        return WCF::getSession()->getPermission("admin.content.label.canManageLabel");
    }

    #[\Override]
    protected function createObjectList(): I18nLabelList
    {
        return new I18nLabelList();
    }

    #[\Override]
    protected function getInitializedEvent(): LabelGridViewInitialized
    {
        return new LabelGridViewInitialized($this);
    }

    /**
     * @return array<int, string>
     */
    protected function getAvailableLabelGroups(): array
    {
        $labelGroupList = new LabelGroupList();
        $labelGroupList->readObjects();
        $labelGroups = \array_map(static fn($group) => $group->getExtendedTitle(), $labelGroupList->getObjects());

        $collator = new \Collator(WCF::getLanguage()->getLocale());
        \uasort(
            $labelGroups,
            static fn(string $groupA, string $groupB) => $collator->compare($groupA, $groupB)
        );

        return $labelGroups;
    }
}
