<?php

namespace wcf\system\form\builder\field;

use wcf\system\WCF;

/**
 * Provides default implementations of `IPlaceholderFormField` methods.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @since   5.2
 */
trait TPlaceholderFormField
{
    /**
     * placeholder value of this element
     * @var ?string
     */
    protected $placeholder;

    /**
     * Returns the placeholder value of this field or `null` if no placeholder has
     * been set.
     *
     * @return ?string
     */
    public function getPlaceholder()
    {
        return $this->placeholder;
    }

    /**
     * Sets the placeholder value of this field using the given language item
     * and returns this element. If `null` is passed, the placeholder value is
     * removed.
     *
     * @param ?string $languageItem language item containing the placeholder or `null` to unset description
     * @param array<string, mixed> $variables additional variables used when resolving the language item
     * @return static this field
     *
     * @throws \InvalidArgumentException if the given value is no or otherwise invalid
     */
    public function placeholder($languageItem = null, array $variables = [])
    {
        if ($languageItem === null) {
            if ($variables !== []) {
                throw new \InvalidArgumentException(
                    "Cannot use variables when unsetting placeholder of field '{$this->getId()}'"
                );
            }

            $this->placeholder = null;
        } else {
            // @phpstan-ignore function.alreadyNarrowedType
            if (!\is_string($languageItem)) {
                throw new \InvalidArgumentException(
                    "Given placeholder language item is no string, " . \gettype($languageItem) . " given for field '{$this->getId()}'."
                );
            }

            $this->placeholder = WCF::getLanguage()->getDynamicVariable($languageItem, $variables);
        }

        return $this;
    }
}
