<?php

namespace wcf\system\form\builder\field;

use wcf\data\IStorableObject;
use wcf\system\form\builder\data\processor\CustomFormDataProcessor;
use wcf\system\form\builder\IFormDocument;

/**
 * Implementation of a form field for selecting map coordinates.
 *
 * @author      Marcel Werk
 * @copyright   2001-2025 WoltLab GmbH
 * @license     GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @since       6.2
 */
final class GoogleMapsFormField extends AbstractFormField implements
    IAttributeFormField,
    IAutoFocusFormField,
    ICssClassFormField,
    IImmutableFormField,
    IPlaceholderFormField
{
    use TInputAttributeFormField {
        getReservedFieldAttributes as private getDefaultReservedFieldAttributes;
    }
    use TAutoFocusFormField;
    use TCssClassFormField;
    use TImmutableFormField;
    use TPlaceholderFormField;

    /**
     * @inheritDoc
     */
    protected $javaScriptDataHandlerModule = 'WoltLabSuite/Core/Form/Builder/Field/GoogleMaps';

    /**
     * @inheritDoc
     */
    protected $templateName = 'shared_googleMapsFormField';

    private float $latitude = 0;
    private float $longitude = 0;

    public function __construct()
    {
        $this->addFieldClass('long');
    }

    /**
     * @return string[]
     */
    protected static function getReservedFieldAttributes(): array
    {
        return \array_merge(
            static::getDefaultReservedFieldAttributes(),
            [
                'data-google-maps-geocoding-store',
                'data-google-maps-geocoding',
                'data-google-maps-marker',
            ]
        );
    }

    #[\Override]
    public function readValue()
    {
        if ($this->getDocument()->hasRequestData($this->getPrefixedId())) {
            $this->value = $this->getDocument()->getRequestData($this->getPrefixedId());
        }

        if ($this->getDocument()->hasRequestData($this->getPrefixedId() . '_coordinates')) {
            $coordinates = explode(',', $this->getDocument()->getRequestData(
                $this->getPrefixedId() . '_coordinates'
            ));
            if (\count($coordinates) === 2) {
                $this->latitude = \floatval($coordinates[0]);
                $this->longitude = \floatval($coordinates[1]);
            }
        }

        return $this;
    }

    #[\Override]
    public function populate()
    {
        parent::populate();

        $this->getDocument()->getDataHandler()->addProcessor(new CustomFormDataProcessor(
            'coordinates',
            function (IFormDocument $document, array $parameters) {
                if ($this->getValue()) {
                    $parameters[$this->getObjectProperty() . '_latitude'] = $this->getLatitude();
                    $parameters[$this->getObjectProperty() . '_longitude'] = $this->getLongitude();
                }

                return $parameters;
            }
        ));

        return $this;
    }

    public function getLatitude(): float
    {
        return $this->latitude;
    }

    public function getLongitude(): float
    {
        return $this->longitude;
    }

    public function coordinates(float $latitude, float $longitude): static
    {
        $this->latitude = $latitude;
        $this->longitude = $longitude;

        return $this;
    }

    #[\Override]
    public function updatedObject(array $data, IStorableObject $object, $loadValues = true): static
    {
        if ($this->isImmutable()) {
            $loadValues = true;
        }

        if (
            $loadValues
            && isset($data[$this->getObjectProperty() . '_latitude'])
            && isset($data[$this->getObjectProperty() . '_longitude'])
        ) {
            $this->coordinates(
                $data[$this->getObjectProperty() . '_latitude'],
                $data[$this->getObjectProperty() . '_longitude']
            );
        }

        return parent::updatedObject($data, $object, $loadValues);
    }
}
