<?php

namespace wcf\system\acp\dashboard\box;

use wcf\system\WCF;

/**
 * ACP dashboard box that shows system information.
 *
 * @author      Marcel Werk
 * @copyright   2001-2023 WoltLab GmbH
 * @license     GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @since       6.1
 * @phpstan-type ServerInfo array{
 *  os: string,
 *  webserver: string,
 *  mySQLVersion: string,
 *  memoryLimit: string,
 *  upload_max_filesize: string,
 *  postMaxSize: string,
 *  innodbFlushLogAtTrxCommit: int,
 * }
 */
final class SystemInfoAcpDashboardBox extends AbstractAcpDashboardBox
{
    #[\Override]
    public function getTitle(): string
    {
        return WCF::getLanguage()->get('wcf.acp.dashboard.box.systemInfo');
    }

    #[\Override]
    public function getContent(): string
    {
        return WCF::getTPL()->render('wcf', 'systemInfoAcpDashboardBox', $this->getVariables());
    }

    #[\Override]
    public function getName(): string
    {
        return 'com.woltlab.wcf.systemInfo';
    }

    /**
     * @return array{
     *  databaseName: string,
     *  server: ServerInfo,
     * }
     */
    private function getVariables(): array
    {
        return [
            'databaseName' => $this->getDatabaseName(),
            'server' => $this->getServerInfo(),
        ];
    }

    private function getDatabaseName(): string
    {
        $sql = "SELECT DATABASE()";
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute();
        return $statement->fetchSingleColumn();
    }

    /**
     * @return ServerInfo
     */
    private function getServerInfo(): array
    {
        return [
            'os' => \PHP_OS,
            'webserver' => $_SERVER['SERVER_SOFTWARE'] ?? '',
            'mySQLVersion' => WCF::getDB()->getVersion(),
            'memoryLimit' => \ini_get('memory_limit'),
            'upload_max_filesize' => \ini_get('upload_max_filesize'),
            'postMaxSize' => \ini_get('post_max_size'),
            'innodbFlushLogAtTrxCommit' => $this->getInnodbFlushLogAtTrxCommit(),
        ];
    }

    private function getInnodbFlushLogAtTrxCommit(): int
    {
        $sql = "SELECT @@innodb_flush_log_at_trx_commit";
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute();
        return $statement->fetchSingleColumn();
    }
}
