<?php

namespace wcf\system\acp\dashboard\box;

use wcf\data\package\PackageCache;
use wcf\system\package\license\LicenseApi;
use wcf\system\package\license\LicenseData;
use wcf\system\WCF;

/**
 * ACP dashboard box listing expired and expiring licenses.
 *
 * @author      Olaf Braun
 * @copyright   2001-2024 WoltLab GmbH
 * @license     GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @since       6.1
 */
final class ExpiringLicensesAcpDashboardBox extends AbstractAcpDashboardBox
{
    private ?LicenseData $licenseData;

    /**
     * @var array<string, int>
     */
    private array $expiredLicenses;

    private bool $fetchLicenseDataFailed = false;

    #[\Override]
    public function isAccessible(): bool
    {
        return WCF::getSession()->getPermission('admin.configuration.package.canEditServer');
    }

    #[\Override]
    public function hasContent(): bool
    {
        return $this->getExpiredLicenses() !== [] || $this->fetchLicenseDataFailed;
    }

    /**
     * @return array<string, int>
     */
    private function getExpiredLicenses(): array
    {
        if (!isset($this->expiredLicenses)) {
            $licenseData = $this->getLicenseData();
            if ($licenseData === null) {
                $this->expiredLicenses = [];
                return $this->expiredLicenses;
            }

            $this->expiredLicenses = \array_filter(
                $licenseData->license['expiryDates'] ?? [],
                function ($date, $packageName) {
                    if ($date === -1) {
                        return false;
                    }

                    $expiryDate = \TIME_NOW + 7_776_000; //90 days
                    if (PackageCache::getInstance()->getPackageID($packageName) === null) {
                        // package not installed
                        return false;
                    }
                    return $date < $expiryDate;
                },
                \ARRAY_FILTER_USE_BOTH
            );
        }
        return $this->expiredLicenses;
    }

    private function getLicenseData(): ?LicenseData
    {
        if (!isset($this->licenseData)) {
            $licenseApi = new LicenseApi();

            try {
                $this->licenseData = $licenseApi->getUpToDateLicenseData();
            } catch (\Throwable) {
                $this->licenseData = null;
                $this->fetchLicenseDataFailed = true;
            }
        }

        return $this->licenseData;
    }

    #[\Override]
    public function getTitle(): string
    {
        return WCF::getLanguage()->get('wcf.acp.dashboard.box.expiringLicenses');
    }

    #[\Override]
    public function getContent(): string
    {
        if ($this->fetchLicenseDataFailed) {
            return \sprintf(
                '<woltlab-core-notice type="error">%s</woltlab-core-notice>',
                WCF::getLanguage()->getDynamicVariable('wcf.acp.license.error.parsingFailed', [
                    'licenseData' => null,
                ])
            );
        }

        $packages = [];
        foreach (\array_keys($this->getExpiredLicenses()) as $packageName) {
            $packages[$packageName] = PackageCache::getInstance()->getPackageByIdentifier($packageName);
        }

        $licenseNo = $this->getLicenseData()->getLicenseNumber();
        if ($licenseNo === null) {
            $ctaLink = 'https://www.woltlab.com/license-list/';
        } else {
            $ctaLink = \sprintf(
                'https://www.woltlab.com/license-extend/%s/',
                $licenseNo,
            );
        }

        return WCF::getTPL()->render('wcf', 'expiringLicensesAcpDashboardBox', [
            'packages' => $packages,
            'expiredLicenses' => \array_filter($this->getExpiredLicenses(), fn($date) => $date < \TIME_NOW),
            'expiringLicenses' => \array_filter($this->getExpiredLicenses(), fn($date) => $date >= \TIME_NOW),
            'ctaLink' => $ctaLink,
        ]);
    }

    #[\Override]
    public function getName(): string
    {
        return 'com.woltlab.wcf.expiringLicenses';
    }
}
