<?php

namespace wcf\page;

use Laminas\Diactoros\Response\RedirectResponse;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectList;
use wcf\data\object\type\ObjectType;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\tag\Tag;
use wcf\system\exception\IllegalLinkException;
use wcf\system\request\LinkHandler;
use wcf\system\tagging\ITaggedListViewProvider;
use wcf\system\tagging\TypedTagCloud;
use wcf\system\WCF;
use wcf\util\StringUtil;

/**
 * Shows the a list of tagged objects.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @deprecated 6.2 Use `TaggedListViewPage` instead.
 *
 * @extends MultipleLinkPage<DatabaseObjectList<DatabaseObject>>
 */
class TaggedPage extends MultipleLinkPage
{
    /**
     * list of available taggable object types
     * @var ObjectType[]
     */
    public $availableObjectTypes = [];

    /**
     * @inheritDoc
     */
    public $neededModules = ['MODULE_TAGGING'];

    /**
     * @inheritDoc
     */
    public $neededPermissions = ['user.tag.canViewTag'];

    /**
     * tag id
     * @var int
     */
    public $tagID = 0;

    /**
     * tag object
     * @var Tag
     */
    public $tag;

    /**
     * object type object
     * @var ?ObjectType
     */
    public $objectType;

    /**
     * tag cloud
     * @var TypedTagCloud
     */
    public $tagCloud;

    /**
     * @var int[]
     * @since 6.0
     */
    public $itemsPerType = [];

    /**
     * @inheritDoc
     */
    public function readParameters()
    {
        parent::readParameters();

        // get tag id
        if (isset($_REQUEST['id'])) {
            $this->tagID = \intval($_REQUEST['id']);
        }
        $this->tag = new Tag($this->tagID);
        if (!$this->tag->tagID) {
            throw new IllegalLinkException();
        }

        // filter taggable object types by options and permissions
        $this->availableObjectTypes = ObjectTypeCache::getInstance()->getObjectTypes('com.woltlab.wcf.tagging.taggableObject');
        foreach ($this->availableObjectTypes as $key => $objectType) {
            if (!$objectType->validateOptions() || !$objectType->validatePermissions()) {
                unset($this->availableObjectTypes[$key]);
            }
        }

        if (empty($this->availableObjectTypes)) {
            throw new IllegalLinkException();
        }

        $this->readItemsPerType();

        // get object type
        if (isset($_REQUEST['objectType'])) {
            $objectType = StringUtil::trim($_REQUEST['objectType']);
            if (!isset($this->availableObjectTypes[$objectType])) {
                throw new IllegalLinkException();
            }
            $this->objectType = $this->availableObjectTypes[$objectType];
        } else {
            foreach ($this->availableObjectTypes as $key => $objectType) {
                if ($this->itemsPerType[$key]) {
                    $this->objectType = $objectType;
                    break;
                }
            }

            if (!$this->objectType) {
                throw new IllegalLinkException();
            }
        }

        if ($this->objectType->getProcessor() instanceof ITaggedListViewProvider) {
            return new RedirectResponse(
                LinkHandler::getInstance()->getControllerLink(TaggedListViewPage::class, [
                    'objectType' => $this->objectType->objectType,
                    'tagIDs' => [$this->tagID],
                ]),
            );
        }
    }

    /**
     * @inheritDoc
     */
    protected function initObjectList()
    {
        $this->objectList = $this->objectType->getProcessor()->getObjectList($this->tag);
    }

    /**
     * @inheritDoc
     */
    public function readData()
    {
        parent::readData();

        $this->tagCloud = new TypedTagCloud($this->objectType->objectType);
    }

    /**
     * @inheritDoc
     */
    public function assignVariables()
    {
        parent::assignVariables();

        WCF::getTPL()->assign([
            'tag' => $this->tag,
            'tags' => $this->tagCloud->getTags(100),
            'availableObjectTypes' => $this->availableObjectTypes,
            'objectType' => $this->objectType->objectType,
            'resultListTemplateName' => $this->objectType->getProcessor()->getTemplateName(),
            'resultListApplication' => $this->objectType->getProcessor()->getApplication(),
            'itemsPerType' => $this->itemsPerType,
            'objectTypeLinks' => $this->getObjectTypeLinks(),
        ]);

        if (\count($this->objectList) === 0) {
            @\header('HTTP/1.1 404 Not Found');
        }
    }

    /**
     * @return list<array{
     *  objectType: string,
     *  title: string,
     *  link: string,
     *  items: int,
     * }>
     */
    protected function getObjectTypeLinks(): array
    {
        $links = [];
        foreach ($this->availableObjectTypes as $objectType) {
            $items = $this->itemsPerType[$objectType->objectType] ?? 0;
            if ($items === 0) {
                continue;
            }

            $processor = $objectType->getProcessor();
            if ($processor instanceof ITaggedListViewProvider) {
                $title = $processor->getObjectTypeTitle();
                $controller = TaggedListViewPage::class;
            } else {
                $title = WCF::getLanguage()->getDynamicVariable('wcf.tagging.objectType.' . $objectType->objectType);
                $controller = CombinedTaggedPage::class;
            }

            $links[] = [
                'objectType' => $objectType->objectType,
                'title' => $title,
                'link' => LinkHandler::getInstance()->getControllerLink(
                    $controller,
                    [
                        'objectType' => $objectType->objectType,
                        'tagIDs' => [$this->tagID]
                    ]
                ),
                'items' => $items,
            ];
        }

        return $links;
    }

    protected function readItemsPerType(): void
    {
        foreach ($this->availableObjectTypes as $key => $objectType) {
            $processor = $objectType->getProcessor();
            if ($processor instanceof ITaggedListViewProvider) {
                $this->itemsPerType[$key] = $processor->getListView([$this->tagID])->countItems();
            } else {
                $objectList = $processor->getObjectList($this->tag);
                \assert($objectList instanceof DatabaseObjectList);
                $this->itemsPerType[$key] = $objectList->countObjects();
            }
        }
    }
}
