<?php

namespace wcf\data\comment;

use wcf\data\like\ILikeObjectTypeProvider;
use wcf\data\like\object\ILikeObject;
use wcf\data\object\type\AbstractObjectTypeProvider;
use wcf\system\cache\runtime\CommentRuntimeCache;
use wcf\system\cache\runtime\ViewableCommentRuntimeCache;
use wcf\system\comment\CommentHandler;
use wcf\system\like\IViewableLikeProvider;

/**
 * Object type provider for comments
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 *
 * @extends AbstractObjectTypeProvider<LikeableComment>
 * @implements ILikeObjectTypeProvider<LikeableComment>
 */
class LikeableCommentProvider extends AbstractObjectTypeProvider implements
    ILikeObjectTypeProvider,
    IViewableLikeProvider
{
    /**
     * @inheritDoc
     */
    public $className = Comment::class;

    /**
     * @inheritDoc
     */
    public $decoratorClassName = LikeableComment::class;

    /**
     * @inheritDoc
     */
    public $listClassName = CommentList::class;

    /**
     * @inheritDoc
     */
    public function checkPermissions(ILikeObject $object)
    {
        \assert($object instanceof LikeableComment);

        if (!$object->commentID) {
            return false;
        }

        $objectType = CommentHandler::getInstance()->getObjectType($object->objectTypeID);

        return CommentHandler::getInstance()->getCommentManager($objectType->objectType)->isAccessible($object->objectID);
    }

    /**
     * @inheritDoc
     */
    public function prepare(array $likes)
    {
        $commentIDs = [];
        foreach ($likes as $like) {
            $commentIDs[] = $like->objectID;
        }

        // fetch comments
        $comments = ViewableCommentRuntimeCache::getInstance()->getObjects($commentIDs);

        // group likes by object type id
        $likeData = [];
        foreach ($likes as $like) {
            if (isset($comments[$like->objectID])) {
                if (!isset($likeData[$comments[$like->objectID]->objectTypeID])) {
                    $likeData[$comments[$like->objectID]->objectTypeID] = [];
                }
                $likeData[$comments[$like->objectID]->objectTypeID][] = $like;
            }
        }

        foreach ($likeData as $objectTypeID => $likes) {
            $objectType = CommentHandler::getInstance()->getObjectType($objectTypeID);
            $commentManager = CommentHandler::getInstance()->getCommentManager($objectType->objectType);
            if ($commentManager instanceof IViewableLikeProvider) {
                $commentManager->prepare($likes);
            }
        }
    }

    #[\Override]
    public function getObjectByID($objectID)
    {
        return new LikeableComment(CommentRuntimeCache::getInstance()->getObject($objectID));
    }

    #[\Override]
    public function getObjectsByIDs(array $objectIDs)
    {
        return \array_map(
            static fn(Comment $comment) => new LikeableComment($comment),
            CommentRuntimeCache::getInstance()->getObjects($objectIDs)
        );
    }
}
