/**
 * Handles bulk interactions that call a RPC endpoint.
 *
 * @author Marcel Werk
 * @copyright 2001-2025 WoltLab GmbH
 * @license GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 * @since 6.2
 */
define(["require", "exports", "WoltLabSuite/Core/Api/DeleteObject", "WoltLabSuite/Core/Api/PostObject", "../Confirmation", "WoltLabSuite/Core/Component/Snackbar"], function (require, exports, DeleteObject_1, PostObject_1, Confirmation_1, Snackbar_1) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.setup = setup;
    async function handleRpcInteraction(container, objectIds, endpoint, label, confirmationType, customConfirmationMessage = "") {
        const confirmationResult = await (0, Confirmation_1.handleConfirmation)("", confirmationType, customConfirmationMessage);
        if (!confirmationResult.result) {
            return;
        }
        const snackbar = (0, Snackbar_1.showProgressSnackbar)(label, objectIds.length);
        for (let i = 0; i < objectIds.length; i++) {
            if (confirmationType == Confirmation_1.ConfirmationType.Delete) {
                await (0, DeleteObject_1.deleteObject)(endpoint.replace(/%s/, objectIds[i].toString()));
            }
            else {
                await (0, PostObject_1.postObject)(endpoint.replace(/%s/, objectIds[i].toString()), confirmationResult.reason ? { reason: confirmationResult.reason } : undefined);
            }
            snackbar.setIteration(i + 1);
            const element = container.querySelector(`[data-object-id="${objectIds[i]}"]`);
            if (!element) {
                continue;
            }
            if (confirmationType == Confirmation_1.ConfirmationType.Delete) {
                element.dispatchEvent(new CustomEvent("interaction:remove", {
                    bubbles: true,
                }));
            }
            else {
                element.dispatchEvent(new CustomEvent("interaction:invalidate", {
                    bubbles: true,
                }));
            }
        }
        snackbar.markAsDone();
        container.dispatchEvent(new CustomEvent("interaction:reset-selection"));
    }
    function setup(identifier, container) {
        container.addEventListener("bulk-interaction", (event) => {
            if (event.detail.bulkInteraction === identifier) {
                void handleRpcInteraction(container, JSON.parse(event.detail.objectIds), event.detail.endpoint, event.detail.label, event.detail.confirmationType, event.detail.confirmationMessage);
            }
        });
    }
});
