const container = document.createElement("div");
container.id = "pageOverlayContainer";

const adoptiveParents: HTMLElement[] = [];

/**
 * Adopts the page overlay container by moving it inside
 * the provided element. Required for dialogs using the
 * native `<dialog>` element to escape the top-level
 * backdrop generated by the browser.
 */
export function adoptPageOverlayContainer(element: HTMLElement): void {
  adoptiveParents.push(element);

  if ("moveBefore" in Element.prototype && document.body.contains(container)) {
    element.moveBefore(container, null);
  } else {
    element.append(container);
  }
}

/**
 * Releases the page overlay container again, allowing
 * it to return to its previous position. Must be called
 * in reverse order to ensure the correct placement.
 */
export function releasePageOverlayContainer(element: HTMLElement): void {
  if (!adoptiveParents.includes(element)) {
    throw new Error("Cannot release the page overlay from a container that (no longer) adopts it.", {
      cause: {
        element,
      },
    });
  }

  const index = adoptiveParents.indexOf(element);
  if (index === 0) {
    throw new Error("Cannot release the page overlay, there is no previous owner.", {
      cause: {
        element,
      },
    });
  }

  if (index === adoptiveParents.length - 1) {
    adoptiveParents.pop();

    const previousParent = adoptiveParents[adoptiveParents.length - 1];
    if ("moveBefore" in Element.prototype) {
      previousParent.moveBefore(container, null);
    } else {
      previousParent.append(container);
    }

    return;
  }

  // `element` was an intermediate owner but is no longer holding a reference
  // to it. This can happen when there are nested dialogs and the parent dialog
  // is disposed while the new dialog is being shown.
  adoptiveParents.splice(index, 1);
}

/**
 * Returns the global page overlay container used for
 * elements that should be displayed out of context,
 * such as tooltips or drop-down menus.
 */
export function getPageOverlayContainer(): HTMLElement {
  return container;
}
