<?php

namespace wcf\acp\form;

use wcf\data\style\Style;
use wcf\data\user\avatar\UserAvatar;
use wcf\data\user\avatar\UserAvatarAction;
use wcf\data\user\cover\photo\UserCoverPhoto;
use wcf\data\user\group\UserGroup;
use wcf\data\user\User;
use wcf\data\user\UserAction;
use wcf\data\user\UserEditor;
use wcf\data\user\UserProfileAction;
use wcf\form\AbstractForm;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\html\upcast\HtmlUpcastProcessor;
use wcf\system\message\embedded\object\MessageEmbeddedObjectManager;
use wcf\system\moderation\queue\ModerationQueueManager;
use wcf\system\option\user\UserOptionHandler;
use wcf\system\style\StyleHandler;
use wcf\system\user\command\SetColorScheme;
use wcf\system\user\multifactor\Setup;
use wcf\system\WCF;
use wcf\util\StringUtil;

/**
 * Shows the user edit form.
 *
 * @author  Marcel Werk
 * @copyright   2001-2020 WoltLab GmbH
 * @license GNU Lesser General Public License <http://opensource.org/licenses/lgpl-license.php>
 */
class UserEditForm extends UserAddForm
{
    /**
     * @inheritDoc
     */
    public $activeMenuItem = 'wcf.acp.menu.link.user.list';

    /**
     * @inheritDoc
     */
    public $neededPermissions = ['admin.user.canEditUser'];

    /**
     * user id
     * @var int
     */
    public $userID = 0;

    /**
     * user editor object
     * @var UserEditor
     */
    public $user;

    /**
     * ban status
     * @var bool
     */
    public $banned = 0;

    /**
     * ban reason
     * @var string
     */
    public $banReason = '';

    /**
     * date when the ban expires
     * @var int
     */
    public $banExpires = 0;

    /**
     * user avatar object
     * @var UserAvatar
     */
    public $userAvatar;

    /**
     * avatar type
     * @var string
     */
    public $avatarType = 'none';

    /**
     * true to disable this avatar
     * @var bool
     */
    public $disableAvatar = 0;

    /**
     * reason
     * @var string
     */
    public $disableAvatarReason = '';

    /**
     * date when the avatar will be enabled again
     * @var int
     */
    public $disableAvatarExpires = 0;

    /**
     * user cover photo object
     * @var UserCoverPhoto
     */
    public $userCoverPhoto;

    /**
     * true to disable this cover photo
     * @var bool
     */
    public $disableCoverPhoto = 0;

    /**
     * reason
     * @var string
     */
    public $disableCoverPhotoReason = '';

    /**
     * date when the cover photo will be enabled again
     * @var int
     */
    public $disableCoverPhotoExpires = 0;

    /**
     * true to delete the current cover photo
     * @var bool
     */
    public $deleteCoverPhoto = 0;

    /**
     * true to delete the current auth data
     * @var bool
     */
    public $disconnect3rdParty = 0;

    /**
     * true to disable multifactor authentication
     * @var bool
     */
    public $multifactorDisable = 0;

    /**
     * list of available styles for the edited user
     * @var         Style[]
     * @since       5.3
     */
    public $availableStyles = [];

    /**
     * id of the used style
     * @var         int
     * @since       5.3
     */
    public $styleID = 0;

    /**
     * @since 6.0
     */
    public string $colorScheme = 'system';

    /**
     * @inheritDoc
     */
    public function readParameters()
    {
        if (isset($_REQUEST['id'])) {
            $this->userID = \intval($_REQUEST['id']);
        }
        $user = new User($this->userID);
        if (!$user->userID) {
            throw new IllegalLinkException();
        }

        $this->user = new UserEditor($user);
        if (!UserGroup::isAccessibleGroup($this->user->getGroupIDs())) {
            throw new PermissionDeniedException();
        }
        $this->attachmentObjectID = $this->user->userID;

        parent::readParameters();
    }

    /**
     * @inheritDoc
     */
    protected function initOptionHandler()
    {
        \assert($this->optionHandler instanceof UserOptionHandler);
        $this->optionHandler->setUser($this->user->getDecoratedObject());
    }

    /**
     * @inheritDoc
     */
    public function readFormParameters()
    {
        parent::readFormParameters();

        if (!WCF::getSession()->getPermission('admin.user.canEditPassword') || !empty($this->user->authData)) {
            $this->password = '';
        }
        if (!WCF::getSession()->getPermission('admin.user.canEditMailAddress')) {
            $this->email = $this->user->email;
        }

        if (!empty($_POST['banned'])) {
            $this->banned = 1;
        }
        if (isset($_POST['banReason'])) {
            $this->banReason = StringUtil::trim($_POST['banReason']);
        }
        if ($this->banned && !isset($_POST['banNeverExpires'])) {
            if (isset($_POST['banExpires'])) {
                $banExpires = \DateTimeImmutable::createFromFormat('!Y-m-d', $_POST['banExpires'], new \DateTimeZone(\TIMEZONE));
                if ($banExpires === false) {
                    $this->banExpires = 0;
                } else {
                    $this->banExpires = $banExpires->getTimestamp();
                }
            }
        }

        if (isset($_POST['avatarType'])) {
            $this->avatarType = $_POST['avatarType'];
        }
        if (isset($_POST['styleID'])) {
            $this->styleID = \intval($_POST['styleID']);
        }
        if (isset($_POST['colorScheme'])) {
            $this->colorScheme = $_POST['colorScheme'];
        }

        if (WCF::getSession()->getPermission('admin.user.canDisableAvatar')) {
            if (!empty($_POST['disableAvatar'])) {
                $this->disableAvatar = 1;
            }
            if (isset($_POST['disableAvatarReason'])) {
                $this->disableAvatarReason = StringUtil::trim($_POST['disableAvatarReason']);
            }
            if ($this->disableAvatar && !isset($_POST['disableAvatarNeverExpires'])) {
                if (isset($_POST['disableAvatarExpires'])) {
                    $this->disableAvatarExpires = @\intval(@\strtotime(StringUtil::trim($_POST['disableAvatarExpires'])));
                }
            }
        }

        if (WCF::getSession()->getPermission('admin.user.canDisableCoverPhoto')) {
            if (isset($_POST['deleteCoverPhoto'])) {
                $this->deleteCoverPhoto = 1;
            }
            if (!empty($_POST['disableCoverPhoto'])) {
                $this->disableCoverPhoto = 1;
            }
            if (isset($_POST['disableCoverPhotoReason'])) {
                $this->disableCoverPhotoReason = StringUtil::trim($_POST['disableCoverPhotoReason']);
            }
            if ($this->disableCoverPhoto && !isset($_POST['disableCoverPhotoNeverExpires'])) {
                if (isset($_POST['disableCoverPhotoExpires'])) {
                    $this->disableCoverPhotoExpires = @\intval(@\strtotime(StringUtil::trim($_POST['disableCoverPhotoExpires'])));
                }
            }
        }

        if (WCF::getSession()->getPermission('admin.user.canEditPassword') && isset($_POST['disconnect3rdParty'])) {
            $this->disconnect3rdParty = 1;
        }
        if (WCF::getSession()->getPermission('admin.user.canEditPassword') && isset($_POST['multifactorDisable'])) {
            $this->multifactorDisable = 1;
        }
    }

    /**
     * @inheritDoc
     */
    public function readData()
    {
        if (empty($_POST)) {
            // get visible languages
            $this->readVisibleLanguages();

            // default values
            $this->readDefaultValues();
        }

        $userProfile = UserProfileRuntimeCache::getInstance()->getObject($this->userID);
        foreach (StyleHandler::getInstance()->getStyles() as $style) {
            if (!$style->isDisabled || $userProfile->getPermission('admin.style.canUseDisabledStyle')) {
                $this->availableStyles[$style->styleID] = $style;
            }
        }

        parent::readData();

        // get the avatar object
        if ($this->avatarType == 'custom' && $this->user->avatarID) {
            $this->userAvatar = new UserAvatar($this->user->avatarID);
        }

        // get the user cover photo object
        if ($this->user->coverPhotoHash) {
            // If the editing user lacks the permissions to view the cover photo, the system
            // will try to load the default cover photo. However, the default cover photo depends
            // on the style, eventually triggering a change to the template group which will
            // fail in the admin panel.
            if ($userProfile->canSeeCoverPhoto()) {
                $this->userCoverPhoto = UserProfileRuntimeCache::getInstance()
                    ->getObject($this->userID)
                    ->getCoverPhoto(true);
            }
        }
    }

    /**
     * Sets the selected languages.
     */
    protected function readVisibleLanguages()
    {
        $this->visibleLanguages = $this->user->getLanguageIDs();
    }

    /**
     * Sets the default values.
     */
    protected function readDefaultValues()
    {
        $this->username = $this->user->username;
        $this->email = $this->user->email;
        $this->groupIDs = $this->user->getGroupIDs(true);
        $this->languageID = $this->user->languageID;
        $this->banned = $this->user->banned;
        $this->banReason = $this->user->banReason;
        $this->banExpires = $this->user->banExpires;
        $this->userTitle = $this->user->userTitle;
        $this->styleID = $this->user->styleID;

        $this->signature = $this->user->signature;
        $this->disableSignature = $this->user->disableSignature;
        $this->disableSignatureReason = $this->user->disableSignatureReason;
        $this->disableSignatureExpires = $this->user->disableSignatureExpires;

        $this->disableAvatar = $this->user->disableAvatar;
        $this->disableAvatarReason = $this->user->disableAvatarReason;
        $this->disableAvatarExpires = $this->user->disableAvatarExpires;

        $this->disableCoverPhoto = $this->user->disableCoverPhoto;
        $this->disableCoverPhotoReason = $this->user->disableCoverPhotoReason;
        $this->disableCoverPhotoExpires = $this->user->disableCoverPhotoExpires;

        if ($this->user->avatarID) {
            $this->avatarType = 'custom';
        }

        $this->colorScheme = $this->user->getUserOption('colorScheme');
    }

    /**
     * @inheritDoc
     */
    public function assignVariables()
    {
        parent::assignVariables();

        $signatureProcessor = new HtmlUpcastProcessor();
        $signatureProcessor->process($this->signature ?: '', 'com.woltlab.wcf.user.signature', $this->user->userID);

        WCF::getTPL()->assign([
            'userID' => $this->user->userID,
            'action' => 'edit',
            'url' => '',
            'markedUsers' => 0,
            'user' => $this->user,
            'banned' => $this->banned,
            'banReason' => $this->banReason,
            'avatarType' => $this->avatarType,
            'disableAvatar' => $this->disableAvatar,
            'disableAvatarReason' => $this->disableAvatarReason,
            'disableAvatarExpires' => $this->disableAvatarExpires,
            'userAvatar' => $this->userAvatar,
            'banExpires' => $this->banExpires,
            'userCoverPhoto' => $this->userCoverPhoto,
            'disableCoverPhoto' => $this->disableCoverPhoto,
            'disableCoverPhotoReason' => $this->disableCoverPhotoReason,
            'disableCoverPhotoExpires' => $this->disableCoverPhotoExpires,
            'deleteCoverPhoto' => $this->deleteCoverPhoto,
            'ownerGroupID' => UserGroup::getOwnerGroupID(),
            'availableStyles' => $this->availableStyles,
            'styleID' => $this->styleID,
            'colorScheme' => $this->colorScheme,
            'signature' => $signatureProcessor->getHtml(),
        ]);
    }

    /**
     * @inheritDoc
     */
    public function save()
    {
        AbstractForm::save();
        $this->htmlInputProcessor->setObjectID($this->userID);
        MessageEmbeddedObjectManager::getInstance()->registerObjects($this->htmlInputProcessor);

        // handle avatar
        if ($this->avatarType != 'custom') {
            // delete custom avatar
            if ($this->user->avatarID) {
                $action = new UserAvatarAction([$this->user->avatarID], 'delete');
                $action->executeAction();
            }
        }

        $avatarData = [];
        if ($this->avatarType === 'none') {
            $avatarData = [
                'avatarID' => null,
            ];
        }

        $this->additionalFields = \array_merge($this->additionalFields, $avatarData);

        if ($this->disconnect3rdParty) {
            $this->additionalFields['authData'] = '';
        }

        // add default groups
        $defaultGroups = UserGroup::getAccessibleGroups([UserGroup::GUESTS, UserGroup::EVERYONE, UserGroup::USERS]);
        $oldGroupIDs = $this->user->getGroupIDs(true);
        foreach ($oldGroupIDs as $oldGroupID) {
            if (isset($defaultGroups[$oldGroupID])) {
                $this->groupIDs[] = $oldGroupID;
            }
        }
        $this->groupIDs = \array_unique($this->groupIDs);

        // save user
        $saveOptions = $this->optionHandler->save();

        $data = [
            'data' => \array_merge($this->additionalFields, [
                'username' => $this->username,
                'email' => $this->email,
                'password' => $this->password,
                'languageID' => $this->languageID,
                'userTitle' => $this->userTitle,
                'signature' => $this->htmlInputProcessor->getHtml(),
                'signatureEnableHtml' => 1,
                'styleID' => $this->styleID,
            ]),
            'groups' => $this->groupIDs,
            'languageIDs' => $this->visibleLanguages,
            'options' => $saveOptions,
            'signatureAttachmentHandler' => $this->attachmentHandler,
        ];
        // handle changed username
        if (\mb_strtolower($this->username) != \mb_strtolower($this->user->username)) {
            $data['data']['lastUsernameChange'] = TIME_NOW;
            $data['data']['oldUsername'] = $this->user->username;
        }

        // handle ban
        if (WCF::getSession()->getPermission('admin.user.canBanUser')) {
            $data['data']['banned'] = $this->banned;
            $data['data']['banReason'] = $this->banReason;
            $data['data']['banExpires'] = $this->banExpires;
        }

        // handle disabled signature
        if (WCF::getSession()->getPermission('admin.user.canDisableSignature')) {
            $data['data']['disableSignature'] = $this->disableSignature;
            $data['data']['disableSignatureReason'] = $this->disableSignatureReason;
            $data['data']['disableSignatureExpires'] = $this->disableSignatureExpires;
        }

        // handle disabled avatar
        if (WCF::getSession()->getPermission('admin.user.canDisableAvatar')) {
            $data['data']['disableAvatar'] = $this->disableAvatar;
            $data['data']['disableAvatarReason'] = $this->disableAvatarReason;
            $data['data']['disableAvatarExpires'] = $this->disableAvatarExpires;
        }

        // handle disabled cover photo
        if (WCF::getSession()->getPermission('admin.user.canDisableCoverPhoto')) {
            $data['data']['disableCoverPhoto'] = $this->disableCoverPhoto;
            $data['data']['disableCoverPhotoReason'] = $this->disableCoverPhotoReason;
            $data['data']['disableCoverPhotoExpires'] = $this->disableCoverPhotoExpires;

            if ($this->deleteCoverPhoto) {
                UserProfileRuntimeCache::getInstance()->getObject($this->userID)->getCoverPhoto()->delete();

                $data['data']['coverPhotoHash'] = null;
                $data['data']['coverPhotoExtension'] = '';

                UserProfileRuntimeCache::getInstance()->removeObject($this->userID);
            }
        }

        $this->objectAction = new UserAction([$this->userID], 'update', $data);
        $this->objectAction->executeAction();

        // disable multifactor authentication
        if (WCF::getSession()->getPermission('admin.user.canEditPassword') && $this->multifactorDisable) {
            WCF::getDB()->beginTransaction();
            $setups = Setup::getAllForUser($this->user->getDecoratedObject());
            foreach ($setups as $setup) {
                $setup->delete();
            }

            $this->user->update([
                'multifactorActive' => 0,
            ]);
            WCF::getDB()->commitTransaction();
        }

        if ($this->user->getUserOption('colorScheme') !== $this->colorScheme) {
            $command = new SetColorScheme($this->user->getDecoratedObject(), $this->colorScheme);
            $command();
        }

        // reload user
        $this->user = new UserEditor(new User($this->userID));

        // update user rank
        if (MODULE_USER_RANK) {
            $action = new UserProfileAction([$this->user], 'updateUserRank');
            $action->executeAction();
        }
        if (MODULE_USERS_ONLINE) {
            $action = new UserProfileAction([$this->user], 'updateUserOnlineMarking');
            $action->executeAction();
        }

        // remove assignments
        $sql = "DELETE FROM wcf" . WCF_N . "_moderation_queue_to_user
                WHERE       userID = ?";
        $statement = WCF::getDB()->prepareStatement($sql);
        $statement->execute([$this->user->userID]);

        // reset moderation count
        ModerationQueueManager::getInstance()->resetModerationCount($this->user->userID);
        $this->saved();

        // reset password
        $this->password = '';

        // reload user
        $this->user = new UserEditor(new User($this->userID));

        // show success message
        WCF::getTPL()->assign('success', true);
    }

    /**
     * @inheritDoc
     */
    protected function validateUsername($username)
    {
        try {
            if (\mb_strtolower($this->user->username) != \mb_strtolower($username)) {
                parent::validateUsername($username);
            }
        } catch (UserInputException $e) {
            if ($e->getField() === 'username' && $e->getType() === 'notUnique') {
                $user2 = User::getUserByUsername($username);
                if ($user2->userID != $this->user->userID) {
                    throw $e;
                }
            } else {
                throw $e;
            }
        }
    }

    #[\Override]
    protected function validateEmail(string $email): void
    {
        if (\mb_strtolower($this->user->email) != \mb_strtolower($email)) {
            parent::validateEmail($email);
        }
    }

    #[\Override]
    protected function validatePassword(
        #[\SensitiveParameter]
        string $password
    ): void {
        if (!empty($password)) {
            parent::validatePassword($password);
        }
    }

    /**
     * Validates the user avatar.
     */
    protected function validateAvatar()
    {
        if ($this->avatarType != 'custom') {
            $this->avatarType = 'none';
        }

        try {
            switch ($this->avatarType) {
                case 'custom':
                    if (!$this->user->avatarID) {
                        throw new UserInputException('customAvatar');
                    }
                    break;
            }
        } catch (UserInputException $e) {
            $this->errorType[$e->getField()] = $e->getType();
        }
    }

    /**
     * @inheritDoc
     */
    public function validate()
    {
        if ($this->user->userID == WCF::getUser()->userID && WCF::getUser()->hasOwnerAccess()) {
            $ownerGroupID = UserGroup::getOwnerGroupID();
            if ($ownerGroupID && !\in_array($ownerGroupID, $this->groupIDs)) {
                // Members of the owner group cannot remove themselves.
                throw new PermissionDeniedException();
            }
        }

        $this->validateAvatar();

        parent::validate();

        if (!isset($this->availableStyles[$this->styleID])) {
            $this->styleID = 0;
        }

        if ($this->colorScheme !== 'light' && $this->colorScheme !== 'dark') {
            $this->colorScheme = 'system';
        }
    }
}
