import type { DataCell } from '@visactor/generate-vchart';
import { isFunction, isValidNumber } from '@visactor/vutils';
import dayjs from 'dayjs';
import customParseFormat from 'dayjs/plugin/customParseFormat';
dayjs.extend(customParseFormat);

export const uniqBy = (array: any, key: string | ((item: any) => string)) => {
  const seen = new Set();
  return array.filter((item: any) => {
    const k = isFunction(key) ? key(item) : item[key];
    if (k === undefined) {
      return false;
    }
    if (!seen.has(k)) {
      seen.add(k);
      return true;
    }
    return false;
  });
};

export const isValidData = (data: DataCell) => {
  return data !== null && data !== undefined && data !== 'null' && data !== '';
};

const dateFormats = [
  {
    key: 'YYYY年MM月DD日',
    regex: /^(\d{4})年(\d{1,2})月(\d{1,2})日$/,
    format: (match: string, year: string, month: string, day: string) =>
      `${year}-${month.padStart(2, '0')}-${day.padStart(2, '0')}`
  },

  {
    key: 'MM月DD日YYYY年',
    regex: /^(\d{1,2})月(\d{1,2})日(\d{4})年$/,
    format: (match: string, month: string, day: string, year: string) =>
      `${year}-${month.padStart(2, '0')}-${year.padStart(2, '0')}`
  },

  {
    key: 'YYYY年MM月',
    regex: /^(\d{4})年(\d{1,2})月$/,
    format: (match: string, year: string, month: string) => `${year}-${month.padStart(2, '0')}`
  },

  { key: 'YYYY年', regex: /^(\d{4})年$/, format: (match: string, year: string) => `${year}` },

  { key: 'MM月', regex: /^(\d{1,2})月$/, format: (match: string, month: string) => `${month.padStart(2, '0')}` },
  {
    key: 'MM月DD日',
    regex: /^(\d{1,2})月(\d{1,2})日$/,
    format: (match: string, month: string, day: string) => `${month.padStart(2, '0')}-${day.padStart(2, '0')}`
  }
];

function convertQuarterToMonth(quarter: string) {
  switch (quarter) {
    case 'Q1':
    case '第一季度':
      return '01-01';
    case 'Q2':
    case '第二季度':
      return '04-01';
    case 'Q3':
    case '第三季度':
      return '07-01';
    case 'Q4':
    case '第四季度':
      return '10-01';
    default:
      throw new Error('Invalid quarter');
  }
}

export function convertStringToDateValue(stringDate: string) {
  const date = `${stringDate}`.trim();
  if (dayjs(date).isValid()) {
    return date;
  }
  const validFormat = dateFormats.find(v => new Date(date.replace(v.regex, v.format)).toString() !== 'Invalid Date');
  if (validFormat && validFormat.regex.test(date)) {
    return date.replace(validFormat.regex, validFormat.format);
  }
  const patterns = [
    {
      regex: /^(\d{4})-Q([1-4])$/,
      format: (match: RegExpMatchArray) => `${match[1]}-${convertQuarterToMonth('Q' + match[2])}`
    },
    {
      regex: /^(\d{4})年第([一二三四])季度$/,
      format: (match: RegExpMatchArray) => `${match[1]}-${convertQuarterToMonth(match[2] + '季度')}`
    },
    { regex: /^Q([1-4])$/, format: (match: RegExpMatchArray) => `${convertQuarterToMonth('Q' + match[1])}` },
    {
      regex: /^(\d{4})年-Q([1-4])$/,
      format: (match: RegExpMatchArray) => `${match[1]}-${convertQuarterToMonth('Q' + match[2])}`
    }
  ];

  for (const pattern of patterns) {
    const match = date.match(pattern.regex);
    if (match) {
      return typeof pattern.format === 'function' ? pattern.format(match) : date.replace(pattern.regex, pattern.format);
    }
  }
  return date;
}

export function average(data: any[], field?: string): number {
  let sum = 0;
  let count = 0;
  data.forEach((x: any) => {
    const v = field ? +x[field] : +x;
    if (isValidNumber(v)) {
      sum += v;
      count++;
    }
  });

  const average = sum / count;
  return average;
}

export function getIntersection(array1: DataCell[], array2: DataCell[]) {
  return array1.filter(value => array2.includes(value));
}

const KNOWLEDGE_START_INDEX = 1;
export const getStrFromArray = (array: string[]) =>
  array.map((item, index) => `${index + KNOWLEDGE_START_INDEX}. ${item}`).join('\n');

// Function to find the last index based on a condition
export function findLastIndex(arr: any[], predicate: (value: any, index: number, array: any[]) => boolean) {
  for (let i = arr.length - 1; i >= 0; i--) {
    if (predicate(arr[i], i, arr)) {
      return i;
    }
  }
  return -1;
}
