import JSON5 from 'json5';
import { isArray } from '@visactor/vutils';
import type { LLMResponse, SimpleFieldInfo } from '../../../../../common/typings';
import type { Parser } from '../../../../../base/tools/parser';
import type { GetQuerySQLContext, GetQuerySQLOutput, SQL } from '../../../../../applications/dataAggregation/types';
import type { Requester } from '../../../../../base/tools/requester';
import { parseGPTResponse, requestGPT } from '../../../../../common/utils/gpt';

type DataQueryResponse = {
  sql: SQL;
  fieldInfo: SimpleFieldInfo[]; //fieldInfo generated by LLM; It may has some new fields after sql query.
  usage: any;
  THOUGHT: string;
};

const parseGPTQueryResponse = (response: string) => {
  const sql = response.match(/sql:\n?```(.*?)```/s)[1];
  const fieldInfoStr = response.match(/fieldInfo:\n?```(.*?)```/s)[1];
  let fieldInfo = [];
  try {
    const tempFieldInfo = JSON5.parse(fieldInfoStr);
    if (isArray(tempFieldInfo)) {
      fieldInfo = tempFieldInfo;
    } else {
      fieldInfo = tempFieldInfo.fieldInfo;
    }
  } catch (e) {
    //fieldInfoStr is not a json string; try to wrap it with []
    fieldInfo = JSON5.parse(`[${fieldInfoStr}]`);
  }
  return {
    sql,
    llmFieldInfo: fieldInfo
  };
};

export const parseDataQueryResponse: Parser<LLMResponse, GetQuerySQLOutput> = (gptResponse: LLMResponse) => {
  const dataQueryResponse: DataQueryResponse = parseGPTResponse(gptResponse);
  const { sql, fieldInfo: responseFiledInfo } = dataQueryResponse;
  if (!sql || !responseFiledInfo) {
    //try to parse the response with another format
    const choices = gptResponse.choices;
    const content = choices[0].message.content;
    return {
      ...parseGPTQueryResponse(content),
      usage: gptResponse.usage
    };
  }
  return { sql, llmFieldInfo: responseFiledInfo, usage: gptResponse.usage };
};

export const dataQueryRequestLLM: Requester<GetQuerySQLContext> = async (
  prompt: string,
  queryDatasetMessage: string,
  context: GetQuerySQLContext
) => {
  const { llmOptions } = context;
  const requestFunc = llmOptions.customRequestFunc?.dataQuery ?? requestGPT;
  const QueryDatasetPrompt = prompt;
  const dataProcessRes = await requestFunc(QueryDatasetPrompt, queryDatasetMessage, llmOptions);
  return dataProcessRes;
};
