/*
  Copyright (C) 2005- 2009, Hammersmith Imanet Ltd
  Copyright (C) 2014, University College London
  This file is part of STIR.

  SPDX-License-Identifier: Apache-2.0

  See STIR/LICENSE.txt for details
*/
/*!
  \file
  \ingroup utilities
  \ingroup scatter
  \brief upsample a coarse scatter estimate and fit it to emission data using a weight/mask projection data

  \see ScatterEstimationByBin::upsample_and_fit_scatter_estimate

  \author Charalampos Tsoumpas
  \author Kris Thielemans

  \par Usage:
  \code
  [--min-scale-factor <number>]
  [--max-scale-factor <number>]
  [--remove-interleaving <1|0>]
  [--half-filter-width <number>]
  --output-filename <filename>
  --data-to-fit <filename>
  --data-to-scale <filename>
  --norm <filename>
  --weights <filename>
  \endcode
  \a remove_interleaving defaults to 1\, \a min-scale-factor to 1e-5,
  and \a max-scale-factor to 1e5.

  The norm parameter expects the filename of a .par file with the following keywords
  \code
  Bin Normalisation parameters:=
  type:= <type info as usual>
  <other keywords for the chosen type>
  END:=
  \endcode
*/

#include "stir/ProjDataInfo.h"
#include "stir/KeyParser.h"
#include "stir/ProjDataInterfile.h"
#include "stir/recon_buildblock/TrivialBinNormalisation.h"
#include "stir/scatter/ScatterEstimation.h"
#include "stir/Succeeded.h"
#include "stir/is_null_ptr.h"
#include <iostream>
#include <string>
/***********************************************************/

static void
print_usage_and_exit(const char* const prog_name)
{
  std::cerr << "\nUsage:\n"
            << prog_name << "\\\n"
            << "\t[--min-scale-factor <number>]\\\n"
            << "\t[--max-scale-factor <number>]\\\n"
            << "\t[--remove-interleaving <1|0>]\\\n"
            << "\t[--half-filter-width <number>]\\\n"
            << "\t--output-filename <filename>\\\n"
            << "\t--data-to-fit <filename>\\\n"
            << "\t--data-to-scale <filename>\\\n"
            << "\t--norm <filename>\\\n"
            << "\t--weights <filename>\n"
            << "remove_interleaving defaults to 1\n"
            << "min-scale-factor to 1e-5, and max scale factor to 1e5\n"
            << "The norm parameter expects the filename of a .par file with the following keywords\n"
            << "Bin Normalisation parameters:=\n"
            << "type:= <type info as usual>\n"
            << "<other keywords for the chosen type>\n"
            << "END:=\n";

  exit(EXIT_FAILURE);
}

int
main(int argc, const char* argv[])
{
  const char* const prog_name = argv[0];

  float min_scale_factor = 1.E-5F;
  float max_scale_factor = 1.E5F;
  bool remove_interleaving = true;
  unsigned int half_filter_width = 1;
  stir::BSpline::BSplineType spline_type = stir::BSpline::linear;
  std::string data_to_fit_filename;
  std::string data_to_scale_filename;
  std::string weights_filename;
  std::string output_filename;
  stir::shared_ptr<stir::BinNormalisation> normalisation_sptr;

  // option processing
  while (argc > 1 && argv[1][1] == '-')
    {
      if (strcmp(argv[1], "--min-scale-factor") == 0)
        {
          min_scale_factor = static_cast<float>(atof(argv[2]));
          argc -= 2;
          argv += 2;
        }
      else if (strcmp(argv[1], "--max-scale-factor") == 0)
        {
          max_scale_factor = static_cast<float>(atof(argv[2]));
          argc -= 2;
          argv += 2;
        }
      else if (strcmp(argv[1], "--BSpline-type") == 0)
        {
          spline_type = (stir::BSpline::BSplineType)atoi(argv[2]);
          argc -= 2;
          argv += 2;
        }
      else if (strcmp(argv[1], "--half-filter-width") == 0)
        {
          half_filter_width = (unsigned)atoi(argv[2]);
          argc -= 2;
          argv += 2;
        }
      else if (strcmp(argv[1], "--remove-interleaving") == 0)
        {
          remove_interleaving = atoi(argv[2]) != 0;
          argc -= 2;
          argv += 2;
        }
      else if (strcmp(argv[1], "--output-filename") == 0)
        {
          output_filename = (argv[2]);
          argc -= 2;
          argv += 2;
        }
      else if (strcmp(argv[1], "--data-to-fit") == 0)
        {
          data_to_fit_filename = argv[2];
          argc -= 2;
          argv += 2;
        }
      else if (strcmp(argv[1], "--data-to-scale") == 0)
        {
          data_to_scale_filename = argv[2];
          argc -= 2;
          argv += 2;
        }
      else if (strcmp(argv[1], "--weights") == 0)
        {
          weights_filename = argv[2];
          argc -= 2;
          argv += 2;
        }
      else if (strcmp(argv[1], "--norm") == 0)
        {
          stir::KeyParser parser;
          parser.add_start_key("Bin Normalisation parameters");
          parser.add_parsing_key("type", &normalisation_sptr);
          parser.add_stop_key("END");
          if (!parser.parse(argv[2]))
            {
              return EXIT_FAILURE;
            }
          argc -= 2;
          argv += 2;
        }
      else
        {
          std::cerr << "\nUnknown option: " << argv[1];
          print_usage_and_exit(prog_name);
        }
    }

  if (argc > 1)
    {
      std::cerr << "Command line should contain only options\n";
      print_usage_and_exit(prog_name);
    }

  if (data_to_fit_filename.size() == 0 || data_to_scale_filename.size() == 0 || weights_filename.size() == 0
      || output_filename.size() == 0)
    {
      std::cerr << "One of the required filenames has not been specified\n";
      print_usage_and_exit(prog_name);
    }

  using stir::ProjData;
  const stir::shared_ptr<ProjData> weights_proj_data_sptr = ProjData::read_from_file(weights_filename);
  const stir::shared_ptr<ProjData> data_to_fit_proj_data_sptr = ProjData::read_from_file(data_to_fit_filename);
  const stir::shared_ptr<ProjData> data_to_scale_proj_data_sptr = ProjData::read_from_file(data_to_scale_filename);

  if (stir::is_null_ptr(normalisation_sptr))
    {
      normalisation_sptr.reset(new stir::TrivialBinNormalisation);
      normalisation_sptr->set_up(data_to_fit_proj_data_sptr->get_exam_info_sptr(),
                                 data_to_fit_proj_data_sptr->get_proj_data_info_sptr()->create_shared_clone());
    }
  stir::shared_ptr<stir::ProjDataInfo> data_to_fit_proj_data_info_sptr
      = data_to_fit_proj_data_sptr->get_proj_data_info_sptr()->create_shared_clone();

  stir::ProjDataInterfile output_proj_data(
      data_to_fit_proj_data_sptr->get_exam_info_sptr(), data_to_fit_proj_data_info_sptr, output_filename);

  stir::ScatterEstimation::upsample_and_fit_scatter_estimate(output_proj_data,
                                                             *data_to_fit_proj_data_sptr,
                                                             *data_to_scale_proj_data_sptr,
                                                             *normalisation_sptr,
                                                             *weights_proj_data_sptr,
                                                             min_scale_factor,
                                                             max_scale_factor,
                                                             half_filter_width,
                                                             spline_type,
                                                             remove_interleaving);
  return EXIT_SUCCESS;
}
