#! /bin/bash
# Shell script for testing lm_to_projdata with ROOT support
# see README.txt
#
#  Copyright (C) 2016, 2017 UCL
#  This file is part of STIR.
#
#  SPDX-License-Identifier: Apache-2.0
#
#  See STIR/LICENSE.txt for details
#      
# Author Nikos Efthimiou, Kris Thielemans

echo This script should work with STIR version 6.x. If you have
echo a later version, you might have to update your test pack.
echo Please check the web site.
echo

#
# Parse option arguments (--)
# Note that the -- is required to suppress interpretation of $1 as options 
# to expr
#
while test `expr -- "$1" : "--.*"` -gt 0
do

if test "$1" = "--help"
  then
    echo "Usage: run_root_GATE.sh [install_dir]"
    echo "See README.txt for more info."
    exit 1
  else
    echo Warning: Unknown option "$1"
    echo rerun with --help for more info.
    exit 1
  fi

  shift 1

done 

if [ $# -eq 1 ]; then
  echo "Prepending $1 to your PATH for the duration of this script."
  PATH=$1:$PATH
fi

command -v lm_to_projdata >/dev/null 2>&1 || { echo "lm_to_projdata not found or not executable. Aborting." >&2; exit 1; }
if command -v root >/dev/null 2>&1
then
  ROOT=root
elif command -v root.exe >/dev/null 2>&1
then
  # if you didn't build ROOT with X, it might not have root, but should have root.exe        
  ROOT=root.exe
else        
  echo "root not found or not executable. Aborting." >&2
  exit 1
fi
echo "Using `command -v ${ROOT}`"

echo "Testing the following executable:"
command -v lm_to_projdata

log=lm_to_projdata_input_formats.log
lm_to_projdata --input-formats  > ${log} 2>&1
if ! grep -q ROOT lm_to_projdata_input_formats.log; then
echo GATE support has not been installed in this system. Aborting.
echo '"lm_to_proj_projdata --input_formats" reported'
cat ${log}
exit 1;
else
echo "GATE support detected!"
fi


echo Executing tests on ROOT files generated by GATE simulations


# first delete any files remaining from a previous run
rm -f my_proj*s lm_to_projdata*.log root*.log

ThereWereErrors=0
export INPUT_ROOT_FILE=test_PET_GATE.root
export INPUT=root_header.hroot
export TEMPLATE=template_for_ROOT_scanner.hs

echo "------------------------------"
echo Running lm_to_projdata for all events

export OUT_PROJDATA_FILE=my_proj_from_lm_all_events
export EXCLUDE_SCATTERED=0
export EXCLUDE_RANDOM=0
log=lm_to_projdata_from_ROOT_all_events.log
lm_to_projdata ./lm_to_projdata.par > ${log} 2>&1 
if [ $? -ne 0 ]; then
  ThereWereErrors=1
  echo "Error running lm_to_projdata."
  echo "CHECK LOG $log"
  error_log_files="${error_log_files} ${log}"
else
  all_events=$(cat ${log}|grep "Number of prompts stored in this time period" | grep -o -E '[0-9]+')

  echo Number of prompts stored in this time period: ${all_events}
fi

echo Reading values directly from ROOT file
log=root_output_all_events.log
${ROOT} -b -l ${INPUT_ROOT_FILE} << EOF >& ${log}
Coincidences->Draw(">>eventlist","","goff");
Int_t N = eventlist->GetN();
cout<<endl<<"Number of prompts stored in this time period:"<< N<<endl;
EOF
if [ $? -ne 0 ]; then
  ThereWereErrors=1
  echo "Error running root."
  echo "CHECK LOG $log"
  error_log_files="${error_log_files} ${log}"
else
  all_root_num=$(cat ${log}| grep "Number of prompts stored in this time period" | grep -o -E '[0-9]+')
  echo All events in ROOT file : ${all_root_num}
fi
if [ "$all_events" != "$all_root_num" ];then
  echo "Mismatched number of events!"
  ThereWereErrors=1
fi
echo "------------------------------"
echo Running lm_to_projdata *ONLY* for true events

export OUT_PROJDATA_FILE=my_proj_from_lm_true_events
export EXCLUDE_SCATTERED=1
export EXCLUDE_RANDOM=1

log=lm_to_projdata_from_ROOT_true_events.log
lm_to_projdata ./lm_to_projdata.par > ${log} 2>&1 
if [ $? -ne 0 ]; then
  ThereWereErrors=1
  echo "Error running lm_to_projdata."
  echo "CHECK LOG $log"
  error_log_files="${error_log_files} ${log}"
else
  true_events=$(cat ${log}| grep "Number of prompts stored in this time period" | grep -o -E '[0-9]+')
  echo Number of prompts stored in this time period: ${true_events}
fi
echo Reading true values from ROOT file ...

log=root_output_true_events.log
${ROOT} -b -l ${INPUT_ROOT_FILE} << EOF >& ${log}
Coincidences->Draw(">>eventlist","eventID1 == eventID2 && comptonPhantom1 == 0 && comptonPhantom2 == 0","goff");
Int_t N = eventlist->GetN();
cout<<endl<<"Number of trues stored in this time period:"<< N<<endl;
EOF
if [ $? -ne 0 ]; then
  ThereWereErrors=1
  echo "Error running root."
  echo "CHECK LOG $log"
  error_log_files="${error_log_files} ${log}"
else
  true_root_num=$(cat ${log}| grep "Number of trues stored in this time period" | grep -o -E '[0-9]+')
  echo True events in ROOT file : ${true_root_num}
fi
if [ "$true_events" != "$true_root_num" ]; then
  echo "Mismatched number of events!"
  ThereWereErrors=1
fi

echo
echo '--------------- End of tests -------------'
echo
if test ${ThereWereErrors} = 1  ; 
then
  echo "Check what went wrong. The following log files might help you:"
  echo "${error_log_files}"
  if [ -n "$TRAVIS" -o -n "$GITHUB_WORKSPACE" ]; then
    # The code runs inside Travis or GHA
    for log in ${error_log_files}; do
      echo "=========== ${log} =========="
      cat ${log}
    done
  fi
else
  echo "Everything seems to be fine !"
  echo 'You could remove all generated files using "rm -f my_* *.log"'
fi

exit ${ThereWereErrors}
