//
/*!

  \file
  \ingroup buildblock
  \brief Implementations for class NonseparableSpatiallyVaryingFilters3D
  
    \author Sanida Mustafovic
    \author Kris Thielemans
    
*/
/*
Copyright (C) 2000- 2003, IRSL
SPDX-License-Identifier: Apache-2.0
See STIR/LICENSE.txt for details
*/
#include "stir_experimental/NonseparableSpatiallyVaryingFilters3D.h"
#include "stir/IndexRange3D.h"
#include "stir/ProjDataFromStream.h"
#include "stir/IO/interfile.h"
#include "stir/CartesianCoordinate3D.h"
#include "stir/RelatedViewgrams.h"
#include "stir_experimental/fft.h"
#include "stir_experimental/ArrayFilter2DUsingConvolution.h"
#include "stir_experimental/recon_buildblock/ProjMatrixByDenselUsingRayTracing.h"
#include "stir/CPUTimer.h"
#include "stir/SegmentByView.h"
#include "stir/include.h"

#include "stir/round.h"
#include <iostream>
#include <fstream>
#include <algorithm>
#include <map>

#ifndef STIR_NO_NAMESPACES
using std::ios;
using std::find;
using std::iostream;
using std::fstream;
using std::cerr;
using std::endl;
using std::map;
#endif

#include "stir_experimental/local_helping_functions.h"
#include "stir_experimental/fwd_and_bck_manipulation_for_SAF.h"



START_NAMESPACE_STIR

void
construct_scaled_filter_coefficients_3D(Array<3,float> &new_filter_coefficients_2D_array,   
					Array<3,float>& kernel_3d,const float kapa0_over_kapa1);
					

//// IMPLEMENTATION /////
/**********************************************************************************************/

void
construct_scaled_filter_coefficients_3D(Array<3,float> &new_filter_coefficients_3D_array,   
					Array<3,float>& kernel_3d,const float kapa0_over_kapa1)
{

  /**************************************************************************************/
  if (kapa0_over_kapa1!=0)
  {
   // in the case where sq_kappas=1 --- scaled_filter == original template filter 
   Array<3,float> filter_coefficients(IndexRange3D(kernel_3d.get_min_index(), kernel_3d.get_max_index(),
   kernel_3d[0].get_min_index(), kernel_3d[0].get_max_index(),
   kernel_3d[0][0].get_min_index(), kernel_3d[0][0].get_max_index()));
    
    filter_coefficients = kernel_3d;
  const int length_of_size_array = 16;
  const float kapa0_over_kapa1_interval_size=10.F;
  static VectorWithOffset<int> size_for_kapa0_over_kapa1;
  if (size_for_kapa0_over_kapa1.get_length()==0)
  {
     size_for_kapa0_over_kapa1.grow(0,length_of_size_array-1);
    size_for_kapa0_over_kapa1.fill(64);
  }
  
  
  float sq_kapas = kapa0_over_kapa1; 
  /******************************************************************************************/
  
  if ( sq_kapas > 10000)
  {
    new_filter_coefficients_3D_array.grow(IndexRange3D(0,0,0,0,0,0));
    new_filter_coefficients_3D_array.fill(1);
  }
  else if (sq_kapas!=1.F)
  {
    const int kapa0_over_kapa1_interval = 
      min(static_cast<int>(floor(kapa0_over_kapa1/kapa0_over_kapa1_interval_size)),
          length_of_size_array-1);

    while(true)
    {
      const int size = size_for_kapa0_over_kapa1[kapa0_over_kapa1_interval];
      const int size_z = kernel_3d.get_length()==1 ? 1 : size;
      const int size_y = size; 
      const int size_x = size;

      info(boost::format("Now doing size %1%") % size);
      
      float inverse_sq_kapas;
      if (fabs((double)sq_kapas ) >0.000000000001)
	inverse_sq_kapas = 1/sq_kapas;
      else 
	inverse_sq_kapas = 0;
      
      static Array<1,float> fft_filter_1D_array_64(1,2*(size_z==1?1:64)*64*64);
      static Array<1,float> fft_filter_1D_array_128(1,2*(size_z==1?1:128)*128*128);
      static Array<1,float> fft_filter_1D_array_256(1,2*(size_z==1?1:256)*256*256);
      //static Array<1,float> fft_filter_1D_array_512(1,2*(size_z==1?1:512)*512*512);
      
      Array<1,float>* fft_filter_1D_array_ptr = 0;
      switch (size)
      {
      case 64:
	fft_filter_1D_array_ptr = &fft_filter_1D_array_64;
	break;
      case 128:
	fft_filter_1D_array_ptr = &fft_filter_1D_array_128;
	break;
      case 256:
	fft_filter_1D_array_ptr = &fft_filter_1D_array_256;
	break;
      //case 512:
	//fft_filter_1D_array_ptr = &fft_filter_1D_array_512;
	//break;
      default:
	error("\nNonseparableSpatiallyVaryingFilters3D: Cannot do this at the moment -- size is too big'.\n");
	break;
      }
      Array<1,float>& fft_filter_1D_array = *fft_filter_1D_array_ptr;
           
      Array<1, int> array_lengths(1,3);
      array_lengths[1] = size_z;
      array_lengths[2] = size_y;
      array_lengths[3] = size_x;
      
      if ( fft_filter_1D_array[1] == 0.F)
      {
	// we have to compute it
	// FIRST PADD 1D FILTER COEFFICIENTS AND MAKE THEM SYMMETRIC 
	// ( DO NOT TAKE IMAGINARY PART INTO ACCOUNT YET)
	/**********************************************************************************/
	
        Array<3,float> filter_coefficients_padded_3D_array(IndexRange3D(1,size_z,1,size_y,1,size_x));

    // here do the padding	
	const int min_kernel_3d_z = kernel_3d.get_min_index();
	const int min_kernel_3d_y = kernel_3d[min_kernel_3d_z].get_min_index();
	const int min_kernel_3d_x = kernel_3d[min_kernel_3d_z][min_kernel_3d_y].get_min_index();
	const int oldsize_z = kernel_3d.get_length();
	const int oldsize_y = kernel_3d[min_kernel_3d_z].get_length();
	const int oldsize_x = kernel_3d[min_kernel_3d_z][min_kernel_3d_y].get_length();

	    const int max_kernel_3d_z= kernel_3d.get_max_index()-min_kernel_3d_z;
	    const int max_kernel_3d_y= kernel_3d[min_kernel_3d_z].get_max_index()-min_kernel_3d_y;
	    const int max_kernel_3d_x= kernel_3d[min_kernel_3d_z][min_kernel_3d_y].get_max_index()-min_kernel_3d_x;
	    for ( int k = -(oldsize_z/2);k<=-(oldsize_z/2)+oldsize_z-1;k++)
	  for ( int j = -(oldsize_y/2);j<=-(oldsize_y/2)+oldsize_y-1;j++)
	    for ( int i = -(oldsize_x/2);i<=-(oldsize_x/2)+oldsize_x-1;i++)
	    {
	      const int newk= k>=0 ? k : size_z+k;
	      const int newj= j>=0 ? j : size_y+j;
	      const int newi= i>=0 ? i : size_x+i;
	      const int oldk= k>=0 ? k : oldsize_z+k;
	      const int oldj= j>=0 ? j : oldsize_y+j;
	      const int oldi= i>=0 ? i : oldsize_x+i;
	      filter_coefficients_padded_3D_array[newk+1][newj+1][newi+1] = 
		kernel_3d[oldk+min_kernel_3d_z][oldj+min_kernel_3d_y][oldi+min_kernel_3d_x];    

	     	    }
#if 0
	    for ( int k = 1; k<=size_z;k++)
	    {
	      for ( int j = 1; j<=size_y;j++)
	      {
		  for ( int i = 1; i<=size_x;i++)
		  {
		    cerr << filter_coefficients_padded_3D_array[k][j][i] << "   " ;

		  }
	
		  cerr << endl;
	      }
	      cerr << endl;
	    } 

#endif	
	 
	// rescale to DC=1
//	filter_coefficients_padded_3D_array /= filter_coefficients_padded_3D_array.sum();
	
	convert_array_3D_into_1D_array(fft_filter_1D_array,filter_coefficients_padded_3D_array);
	//cerr << fft_filter_1D_array[fft_filter_1D_array.get_min_index()]<< "   " << fft_filter_1D_array[fft_filter_1D_array.get_min_index()+1]<<"    ";

      	fourn(fft_filter_1D_array, array_lengths, 3,1);
	fft_filter_1D_array /= sqrt(static_cast<float>(size_z *size_y*size_x));   
      }	

      //for ( int i = fft_filter_1D_array.get_min_index(); i<=fft_filter_1D_array.get_max_index();i++)
	//cerr << fft_filter_1D_array[i]<< "     ";

    
      Array<3,float> new_filter_coefficients_3D_array_tmp (IndexRange3D(1,size_z,1,size_y,1,size_x));           
      
      // WARNING  -- this only works for the FFT where the convention is that the final result
      // obtained from the FFT is divided with sqrt(N*N*N)   
      // initialise to 0 to prevent from warnings
      //fourn(fft_1_1D_array, array_lengths, 3,1); 
      float  fft_1_1D_array = 1/sqrt(static_cast<float>(size_z*size_y*size_x)); 
      {        
	Array<1,float> fft_filter_num_1D_array = fft_filter_1D_array;	
	fft_filter_num_1D_array *= fft_1_1D_array;	
	// TODO we make a copy for the denominator here, which isn't necessary
	Array<1,float> fft_filter_denom_1D_array = fft_filter_1D_array;	
	fft_filter_denom_1D_array*= (sq_kapas-1);
	// add fft of 1 (but that's a real constant)
	for ( int i = fft_filter_1D_array.get_min_index(); i<=fft_filter_1D_array.get_max_index(); i+=2)
	{
	  fft_filter_denom_1D_array[i] += fft_1_1D_array;
	}
	fft_filter_denom_1D_array /= sq_kapas;

     
	divide_complex_arrays(fft_filter_num_1D_array,fft_filter_denom_1D_array);  
	fourn(fft_filter_num_1D_array, array_lengths, 3,-1);
	
	
	// make it consistent with mathemematica -- the output of the       
	fft_filter_num_1D_array  /= sqrt(static_cast<double>(size_z *size_y*size_x));

	
	
	// take the real part only 
	/*********************************************************************************/
	{
	  Array<1,float> real_div_1D_array(1,size_z *size_y*size_x);
	  
	  for (int i=0;i<=(size_z *size_y*size_x)-1;i++)
	    real_div_1D_array[i+1] = fft_filter_num_1D_array[2*i+1];
	  
	  /*********************************************************************************/
	 	  
	  
	  convert_array_1D_into_3D_array(new_filter_coefficients_3D_array_tmp,real_div_1D_array);
	 


	}
      }
      	    
	  int kernel_length_x=0;
	  int kernel_length_y=0;
	  int kernel_length_z=0;
	  
	  // to prevent form aliasing limit the new range for the coefficients to 
	  // filter_coefficients_padded.get_length()/4
	  
	  int threshold = 10000000;
	  // do the x -direction first -- fix y and z to the min and look for the max index in x
	  int kx = new_filter_coefficients_3D_array_tmp.get_min_index();
	  int jx = new_filter_coefficients_3D_array_tmp[kx].get_min_index();
	  for (int i=new_filter_coefficients_3D_array_tmp[kx][jx].get_min_index();
	       i<=new_filter_coefficients_3D_array_tmp[kx][jx].get_max_index()/2;i++)
	  {
	    if (fabs((double)new_filter_coefficients_3D_array_tmp[kx][jx][i])
	      <= new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()].get_min_index()][new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()].get_min_index()].get_min_index()]*1/threshold) break;
	    else (kernel_length_x)++;
	  }
	  
	  /******************************* Y DIRECTION ************************************/
	  
	  
	  int ky = new_filter_coefficients_3D_array_tmp.get_min_index();
	  int iy = new_filter_coefficients_3D_array_tmp[ky][new_filter_coefficients_3D_array_tmp.get_min_index()].get_min_index();
	  for (int j=new_filter_coefficients_3D_array_tmp[ky].get_min_index();j<=new_filter_coefficients_3D_array_tmp[ky].get_max_index()/2;j++)
	  {
	    if (fabs((double)new_filter_coefficients_3D_array_tmp[ky][j][iy])
	      //= new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()]*1/100000000) break;
	      <= new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()].get_min_index()][new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()].get_min_index()].get_min_index()]*1/threshold) break;
	    else (kernel_length_y)++;
	  }
	  
	  /********************************************************************************/
	  
	  /******************************* z DIRECTION ************************************/
	  
	 if (size_z==1)
	   kernel_length_z=1;
	 else
	 {
	    int jz = new_filter_coefficients_3D_array_tmp.get_min_index();
	    int iz = new_filter_coefficients_3D_array_tmp[jz][new_filter_coefficients_3D_array_tmp.get_min_index()].get_min_index();
	    for (int k=new_filter_coefficients_3D_array_tmp.get_min_index();k<=new_filter_coefficients_3D_array_tmp.get_max_index()/2;k++)
	    {
	      if (fabs((double)new_filter_coefficients_3D_array_tmp[k][jz][iz])
		//<= new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()]*1/100000000) break;
		<= new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()].get_min_index()][new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()].get_min_index()].get_min_index()]*1/threshold) break;
	      else (kernel_length_z)++;
	    }
	 }
	  /********************************************************************************/
	  
	  if (kernel_length_x == size_x/2)
	  {
	    warning("NonseparableSpatiallyVaryingFilters3D: kernel_length_x reached maximum length %d. "
	      "First filter coefficient %g, last %g, kappa0_over_kappa1 was %g\n"
	      "Increasing length of FFT array to resolve this problem\n",
	      kernel_length_x, new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()], new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()][kernel_length_x],
	      kapa0_over_kapa1);
	    size_for_kapa0_over_kapa1[kapa0_over_kapa1_interval]*=2;
	    for (int i=kapa0_over_kapa1_interval+1; i<size_for_kapa0_over_kapa1.get_length(); ++i)
	      size_for_kapa0_over_kapa1[i]=
	      max(size_for_kapa0_over_kapa1[i], size_for_kapa0_over_kapa1[kapa0_over_kapa1_interval]);
	  }
	  else if (kernel_length_y == size_y/2)
	  {
	    warning("NonseparableSpatiallyVaryingFilters3D: kernel_length_y reached maximum length %d. "
	      "First filter coefficient %g, last %g, kappa0_over_kappa1 was %g\n"
	      "Increasing length of FFT array to resolve this problem\n",
	      kernel_length_y, new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()], new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][kernel_length_y][new_filter_coefficients_3D_array_tmp.get_min_index()],
	      kapa0_over_kapa1);
	    size_for_kapa0_over_kapa1[kapa0_over_kapa1_interval]*=2;
	    for (int i=kapa0_over_kapa1_interval+1; i<size_for_kapa0_over_kapa1.get_length(); ++i)
	      size_for_kapa0_over_kapa1[i]=
	      max(size_for_kapa0_over_kapa1[i], size_for_kapa0_over_kapa1[kapa0_over_kapa1_interval]);
	  }
	  else if (kernel_length_z == size_z/2)
	  {
	    warning("NonseparableSpatiallyVaryingFilters3D: kernel_length_z reached maximum length %d. "
	      "First filter coefficient %g, last %g, kappa0_over_kappa1 was %g\n"
	      "Increasing length of FFT array to resolve this problem\n",
	      kernel_length_z, 
	      new_filter_coefficients_3D_array_tmp[new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()], new_filter_coefficients_3D_array_tmp[kernel_length_z][new_filter_coefficients_3D_array_tmp.get_min_index()][new_filter_coefficients_3D_array_tmp.get_min_index()],
	      kapa0_over_kapa1);
	    size_for_kapa0_over_kapa1[kapa0_over_kapa1_interval]*=2;
	    for (int i=kapa0_over_kapa1_interval+1; i<size_for_kapa0_over_kapa1.get_length(); ++i)
	      size_for_kapa0_over_kapa1[i]=
	      max(size_for_kapa0_over_kapa1[i], size_for_kapa0_over_kapa1[kapa0_over_kapa1_interval]);
	  }
	  else
	  {	 
	    info(boost::format("kernel lengths found: (z,y,x): (%1%,%2%,%3%)") % kernel_length_z % kernel_length_y % kernel_length_x);
	    new_filter_coefficients_3D_array.grow(IndexRange3D(-(kernel_length_z-1),kernel_length_z-1,
	      -(kernel_length_y-1),kernel_length_y-1,
	      -(kernel_length_x-1),kernel_length_x-1));

	    
	    for (int  k = 0;k<= kernel_length_z-1;k++)
	      for (int  j = 0;j<= kernel_length_y-1;j++)	  
		for (int  i = 0;i<= kernel_length_x-1;i++)	  
		{
		  new_filter_coefficients_3D_array[k][j][i]=
		    new_filter_coefficients_3D_array[k][j][-i]=
		    new_filter_coefficients_3D_array[k][-j][-i]=
		    new_filter_coefficients_3D_array[k][-j][i]=
		    new_filter_coefficients_3D_array[-k][j][i]=
		    new_filter_coefficients_3D_array[-k][-j][i]=
		    new_filter_coefficients_3D_array[-k][j][-i]=
		    new_filter_coefficients_3D_array[-k][-j][-i]=
		    new_filter_coefficients_3D_array_tmp[k+1][j+1][i+1];		  
		}
	    
  	break; // out of while(true)
	  }
    } // this bracket is for the while loop
    }
    else //sq_kappas == 1
    {
      const unsigned int size_x = filter_coefficients[0][0].get_length();  
      const unsigned int size_y = filter_coefficients[0].get_length();  
      const unsigned int size_z = filter_coefficients.get_length();  
      
      const int min_index_z = -(size_z/2);  
      const int min_index_y = -(size_y/2);
      const int min_index_x = -(size_x/2);
      
      new_filter_coefficients_3D_array.grow(IndexRange3D(min_index_z, min_index_z + size_z - 1,
	min_index_y, min_index_y + size_y - 1,
	min_index_x, min_index_x + size_x - 1 ));
      
      for (int  k = 0;k<= new_filter_coefficients_3D_array.get_max_index();k++)
      {
	const int oldk = filter_coefficients.get_min_index()+k;
	for (int  j = 0;j<= new_filter_coefficients_3D_array[oldk].get_max_index();j++)
	{
	  const int oldj = filter_coefficients[oldk].get_min_index()+j;
	  for (int  i = 0;i<= new_filter_coefficients_3D_array[oldk][oldj].get_max_index();i++)	  		  
	  {
	    const int oldi = filter_coefficients[oldk][oldj].get_min_index()+i;
	    const float tmp = filter_coefficients[oldk][oldj][oldi];
	    new_filter_coefficients_3D_array[k][j][i]=
	      new_filter_coefficients_3D_array[k][j][-i]=		    
	      new_filter_coefficients_3D_array[k][-j][i]=
	      new_filter_coefficients_3D_array[k][-j][-i]=
	      new_filter_coefficients_3D_array[-k][j][i]=
	      new_filter_coefficients_3D_array[-k][-j][i]=
	      new_filter_coefficients_3D_array[-k][j][-i]=
	      new_filter_coefficients_3D_array[-k][-j][-i]=
	      tmp;	      
	  }
	}
      }
      
      
    }

#if 0
	  for (int  k = 0;k<= 20;k++)
	      for (int  j = 0;j<= 20;j++)	  
		for (int  i = 0;i<= 20;i++)
		{
		  cerr << new_filter_coefficients_3D_array[k][j][i] << "   " ;
		}

 
#endif
    // rescale to DC=1
    new_filter_coefficients_3D_array /= new_filter_coefficients_3D_array.sum();	
   
}
}



#if 1


template <typename elemT>
NonseparableSpatiallyVaryingFilters3D<elemT>::
NonseparableSpatiallyVaryingFilters3D()
{ 
  set_defaults();
}


template <typename elemT>
NonseparableSpatiallyVaryingFilters3D<elemT>::
NonseparableSpatiallyVaryingFilters3D(string proj_data_filename_v,
				       string attenuation_proj_data_filename_v,
				       const Array<3,float>& filter_coefficients_v,
				       shared_ptr<ProjData> proj_data_ptr_v,
				       shared_ptr<ProjData> attenuation_proj_data_ptr_v,
				       DiscretisedDensity<3,float>* initial_image_v,
				       DiscretisedDensity<3,float>* sensitivity_image_v,
				       DiscretisedDensity<3,float>* precomputed_coefficients_image_v,
				       DiscretisedDensity<3,float>* normalised_bck_image_v,
				       int mask_size_v,  float rescaling_coefficient_v)
				       
				       
{
  assert(filter_coefficients.get_length() == 0 ||
    filter_coefficients.begin()==0);
  
  for (int i = filter_coefficients_v.get_min_index();i<=filter_coefficients_v.get_max_index();i++)
    filter_coefficients[i] = filter_coefficients_v[i];
  proj_data_filename  = proj_data_filename_v;
  attenuation_proj_data_filename = attenuation_proj_data_filename_v;
  proj_data_ptr = proj_data_ptr_v;
  attenuation_proj_data_ptr = attenuation_proj_data_ptr_v;
  initial_image = initial_image_v;
  sensitivity_image = sensitivity_image_v;
  precomputed_coefficients_image = precomputed_coefficients_image_v;
  normalised_bck_image = normalised_bck_image_v;
  mask_size= mask_size_v;
  //num_dim = num_dim_v;
  rescaling_coefficient=rescaling_coefficient_v;
}


template <typename elemT>
Succeeded 
NonseparableSpatiallyVaryingFilters3D<elemT>::
virtual_set_up(const DiscretisedDensity<3,elemT>& density)
{
  proj_data_ptr = 
    ProjData::read_from_file( proj_data_filename); 
  
  if (attenuation_proj_data_filename !="1")
    attenuation_proj_data_ptr =
    ProjData::read_from_file(attenuation_proj_data_filename); 
  else 
    attenuation_proj_data_ptr = NULL;
  
  if (initial_image_filename !="1")
    initial_image =
    DiscretisedDensity<3,float>::read_from_file(initial_image_filename); 
  else 
    initial_image  = NULL;
  
  if (sensitivity_image_filename !="1")
    sensitivity_image =
    DiscretisedDensity<3,float>::read_from_file(sensitivity_image_filename); 
  else 
    sensitivity_image = NULL;
  
  if (precomputed_coefficients_filename !="1")
    precomputed_coefficients_image = 
    DiscretisedDensity<3,float>::read_from_file(precomputed_coefficients_filename);
  else
    precomputed_coefficients_image =NULL; 
  
  if (normalised_bck_filename !="1")
    normalised_bck_image = 
    DiscretisedDensity<3,float>::read_from_file(normalised_bck_filename);
  else
    normalised_bck_image =NULL; 
  
  
  
  return Succeeded::yes;
  
}




template <typename elemT>
void 
NonseparableSpatiallyVaryingFilters3D<elemT>::precalculate_filter_coefficients_3D (VectorWithOffset < VectorWithOffset < VectorWithOffset <shared_ptr <ArrayFilter3DUsingConvolution <float> >  > > >& all_filter_coefficients,
									      DiscretisedDensity<3,elemT>* in_density) const
{

 
  VectorWithOffset < shared_ptr <ArrayFilter3DUsingConvolution <float> > > filter_lookup;
  filter_lookup.grow(1,500);
  const int k_min =1;
  const float k_interval = 0.01F; //0.01F;
  

  shared_ptr<ProjDataInfo> new_data_info_ptr  = proj_data_ptr->get_proj_data_info_sptr()->clone();
  VoxelsOnCartesianGrid<float>* in_density_cast =
    dynamic_cast< VoxelsOnCartesianGrid<float>* >(in_density); 
  

  VoxelsOnCartesianGrid<float> *  vox_image_ptr_1 =
    new VoxelsOnCartesianGrid<float> (IndexRange3D(in_density_cast->get_min_z(),in_density_cast->get_max_z(),
    in_density_cast->get_min_y(),in_density_cast->get_max_y(),
    in_density_cast->get_min_x(),in_density_cast->get_max_x()),
    in_density_cast->get_origin(),in_density_cast->get_voxel_size());  
  
  int start_segment_num = proj_data_ptr->get_min_segment_num();
  int end_segment_num = proj_data_ptr->get_max_segment_num();
  
  VectorWithOffset<SegmentByView<float> *> all_segments(start_segment_num, end_segment_num);
  VectorWithOffset<SegmentByView<float> *> all_segments_for_kappa0(start_segment_num, end_segment_num);   
  VectorWithOffset<SegmentByView<float> *> all_attenuation_segments(start_segment_num, end_segment_num);
  
  
  // first initialise to false
  bool do_attenuation = false;

  for (int segment_num = start_segment_num; segment_num <= end_segment_num; ++segment_num)
  {
    all_segments[segment_num] = new SegmentByView<float>(proj_data_ptr->get_empty_segment_by_view(segment_num));
    all_segments_for_kappa0[segment_num] = new SegmentByView<float>(proj_data_ptr->get_empty_segment_by_view(segment_num));
    
    if (attenuation_proj_data_filename!="1")
    {
      do_attenuation = true;
      all_attenuation_segments[segment_num] = 
	new SegmentByView<float>(attenuation_proj_data_ptr->get_segment_by_view(segment_num));
    }
    else 
    {
      do_attenuation = false;
      all_attenuation_segments[segment_num] = new SegmentByView<float>(proj_data_ptr->get_empty_segment_by_view(segment_num));		 
      (*all_attenuation_segments[segment_num]).fill(1);
    }
    
  }
  
  vox_image_ptr_1->set_origin(Coordinate3D<float>(0,0,0));   
  
  shared_ptr<DiscretisedDensity<3,float> > image_sptr =  vox_image_ptr_1;
  
  shared_ptr<ProjMatrixByDensel> proj_matrix_ptr = 
    new ProjMatrixByDenselUsingRayTracing;
  
  proj_matrix_ptr->set_up(proj_data_ptr->get_proj_data_info_sptr()->clone(),
    image_sptr);
  info(boost::format("%1%") % proj_matrix_ptr->parameter_info());
  
  fwd_densels_all(all_segments,proj_matrix_ptr, proj_data_ptr,
    in_density_cast->get_min_z(), in_density_cast->get_max_z(),
    in_density_cast->get_min_y(), in_density_cast->get_max_y(),
    in_density_cast->get_min_x(), in_density_cast->get_max_x(),
    *in_density);
  
  VoxelsOnCartesianGrid<float> *  vox_image_ptr_kappa0 =
    new VoxelsOnCartesianGrid<float>(IndexRange3D(in_density_cast->get_min_z(),in_density_cast->get_max_z(),
    in_density_cast->get_min_y(),in_density_cast->get_max_y(),
    in_density_cast->get_min_x(),in_density_cast->get_max_x()),
    in_density_cast->get_origin(),in_density_cast->get_voxel_size());  
  
  VoxelsOnCartesianGrid<float> *  vox_image_ptr_kappa1 =
    new VoxelsOnCartesianGrid<float>(IndexRange3D(in_density_cast->get_min_z(),in_density_cast->get_max_z(),
    in_density_cast->get_min_y(),in_density_cast->get_max_y(),
    in_density_cast->get_min_x(),in_density_cast->get_max_x()),
    in_density_cast->get_origin(),in_density_cast->get_voxel_size());  

   VoxelsOnCartesianGrid<float> *  kappa_coefficients =
    new VoxelsOnCartesianGrid<float>(IndexRange3D(in_density_cast->get_min_z(),in_density_cast->get_max_z(),
    in_density_cast->get_min_y(),in_density_cast->get_max_y(),
    in_density_cast->get_min_x(),in_density_cast->get_max_x()),
    in_density_cast->get_origin(),in_density_cast->get_voxel_size());
  
  
  shared_ptr<DiscretisedDensity<3,float> > kappa0_ptr_bck =  vox_image_ptr_kappa0;       
  shared_ptr<DiscretisedDensity<3,float> > kappa1_ptr_bck =  vox_image_ptr_kappa1;   
  
  // WARNING - find a way of finding max in the sinogram
  // TODO - include other segments as well
  float max_in_viewgram =0.F;
  
  for (int segment_num = start_segment_num; segment_num<= end_segment_num; segment_num++) 
  {
    SegmentByView<float> segment_by_view = 
      proj_data_ptr->get_segment_by_view(segment_num);
    const float current_max_in_viewgram = segment_by_view.find_max();
    if ( current_max_in_viewgram >= max_in_viewgram)
      max_in_viewgram = current_max_in_viewgram ;
    else
      continue;
  }
  const float threshold = 0.0001F*max_in_viewgram;  
  
  info(boost::format(" THRESHOLD IS %1%") % threshold);
  
  find_inverse_and_bck_densels(*kappa1_ptr_bck,all_segments,
    all_attenuation_segments,
    vox_image_ptr_kappa1->get_min_z(),vox_image_ptr_kappa1->get_max_z(),
    vox_image_ptr_kappa1->get_min_y(),vox_image_ptr_kappa1->get_max_y(),
    vox_image_ptr_kappa1->get_min_x(),vox_image_ptr_kappa1->get_max_x(),
    *proj_matrix_ptr, do_attenuation,threshold, false); //true);
  
  for (int segment_num = start_segment_num; segment_num <= end_segment_num; ++segment_num)
  { 
    delete all_segments[segment_num];
    delete all_attenuation_segments[segment_num];
  }   
  
  info(boost::format("min and max in image - kappa1 %1%, %2%") % kappa1_ptr_bck->find_min() % kappa1_ptr_bck->find_max());
  
  for (int k=in_density_cast->get_min_z();k<=in_density_cast->get_max_z();k++)   
    for (int j =in_density_cast->get_min_y();j<=in_density_cast->get_max_y();j++)
      for (int i =in_density_cast->get_min_x();i<=in_density_cast->get_max_x();i++)	
      {
	
	// WARNING - only works for segment zero at the moment
	// do the calculation of kappa0 here
	kappa0_ptr_bck->fill(0); 
	for (int segment_num = start_segment_num; 
	segment_num <= end_segment_num; ++segment_num)
	{  
	  (*all_segments_for_kappa0[segment_num]).fill(0);	    
	}
	if (true) //attenuation_proj_data_filename !="1")
	{
	   shared_ptr< VoxelsOnCartesianGrid<float> > in_density_cast_tmp =
	   new VoxelsOnCartesianGrid<float>
	    (IndexRange3D(-mask_size+(ceil(in_density_cast->get_max_z()-in_density_cast->get_min_z())/2),
	     mask_size+(ceil(in_density_cast->get_max_z()-in_density_cast->get_min_z())/2),
	    -mask_size+6,mask_size+6,
	    -mask_size+6,mask_size+6),in_density_cast->get_origin(),in_density_cast->get_voxel_size());  
	 /*shared_ptr< VoxelsOnCartesianGrid<float> > in_density_cast_tmp =
	    new VoxelsOnCartesianGrid<float>(IndexRange3D(k,k,
	      //-mask_size+k,mask_size+k,
	    -mask_size+6,mask_size+6,
	    -mask_size+6,mask_size+6),in_density_cast->get_origin(),in_density_cast->get_voxel_size());  */
	  CPUTimer timer;
	  timer.start();
	  
	  	// SM 23/05/2002 mask now 3D
	  const int min_k = max(in_density_cast->get_min_z(),k-mask_size);
	  const int max_k = min(in_density_cast->get_max_z(),k+mask_size);			
	  const int min_j = max(in_density_cast->get_min_y(),j-mask_size);
	  const int max_j = min(in_density_cast->get_max_y(),j+mask_size);
	  const int min_i = max(in_density_cast->get_min_x(),i-mask_size);
	  const int max_i = min(in_density_cast->get_max_x(),i+mask_size);
	  
	  // the mask size is in 2D only
	  // SM mask now 3D 
	 for (int k_in =min_k;k_in<=max_k;k_in++)
	  for (int j_in =min_j;j_in<=max_j;j_in++)
	    for (int i_in =min_i;i_in<=max_i;i_in++)	
	    {
	      (*in_density_cast_tmp)[k_in-k+(ceil(in_density_cast->get_max_z()-in_density_cast->get_min_z())/2)][j_in-j +6][i_in-i+6] = (*in_density_cast)[k_in][j_in][i_in];
	      //(*in_density_cast_tmp)[k][j_in-j +6][i_in-i+6] = (*in_density_cast)[k][j_in][i_in];
	    }
	    
	    fwd_densels_all(all_segments_for_kappa0,proj_matrix_ptr, proj_data_ptr,
	      in_density_cast_tmp->get_min_z(), in_density_cast_tmp->get_max_z(),
	      in_density_cast_tmp->get_min_y(),in_density_cast_tmp->get_max_y(),
	      in_density_cast_tmp->get_min_x(),in_density_cast_tmp->get_max_x(),
	      *in_density_cast_tmp);
	    
	    find_inverse_and_bck_densels(*kappa0_ptr_bck,all_segments_for_kappa0,
	      all_attenuation_segments,
	     //k,k,
	      (ceil(vox_image_ptr_kappa1->get_max_z()-vox_image_ptr_kappa1->get_min_z()))/2,ceil((vox_image_ptr_kappa1->get_max_z()-vox_image_ptr_kappa1->get_min_z())/2),
	      //0,0,0,0,
	      6,6,6,6,
	      *proj_matrix_ptr,false,threshold, false) ;//true);	  
	    (*kappa0_ptr_bck)[k][j][i] = (*kappa0_ptr_bck)[(ceil(vox_image_ptr_kappa1->get_max_z()-vox_image_ptr_kappa1->get_min_z()))/2][6][6];
	    //(*kappa0_ptr_bck)[k][j][i] = (*kappa0_ptr_bck)[k][6][6];
	    
	    timer.stop();
	    //cerr << "kappa0 time "<< timer.value() << endl;
	}
	else
	{
	  const int min_j = max(in_density_cast->get_min_y(),j-mask_size);
	  const int max_j = min(in_density_cast->get_max_y(),j+mask_size);
	  const int min_i = max(in_density_cast->get_min_x(),i-mask_size);
	  const int max_i = min(in_density_cast->get_max_x(),i+mask_size);
	  
	  fwd_densels_all(all_segments_for_kappa0,proj_matrix_ptr, proj_data_ptr,
	    in_density_cast->get_min_z(), in_density_cast->get_max_z(),
	    min_j,max_j,
	    min_i,max_i,
	    //j-2,j+2,
	    //i-2,i+2,
	    *in_density_cast);
	  
	  find_inverse_and_bck_densels(*kappa0_ptr_bck,all_segments_for_kappa0,
	    all_attenuation_segments,
	    vox_image_ptr_kappa1->get_min_z(),vox_image_ptr_kappa1->get_max_z(),
	    j,j,i,i,
	    *proj_matrix_ptr,false,threshold, true);
	  
	}
	float sq_kapas;
//	float multiply_with_sensitivity;
	if ( fabs((double)(*kappa1_ptr_bck)[k][j][i]) > 0.00000000000001 && 
	  fabs((double)(*kappa0_ptr_bck)[k][j][i]) > 0.00000000000001 )
	{ 
	  sq_kapas =((*kappa0_ptr_bck)[k][j][i]*(*kappa0_ptr_bck)[k][j][i])/((*kappa1_ptr_bck)[k][j][i]*(*kappa1_ptr_bck)[k][j][i]);
	  // cerr << "sq_kapas "  << sq_kapas << endl;

	  (*kappa_coefficients)[k][j][i] = sq_kapas;

	   int k_index ;
	   k_index = round(((float)sq_kapas- k_min)/k_interval);
	   if (k_index < 1) 
	   {k_index = 1;}
	    
	   if ( k_index > 500)
	   { k_index  = 500;}
	  
	  
	  if ( filter_lookup[k_index]==NULL )
	  {
	    Array <3,float> new_coeffs;
	    info(boost::format("computing new filter for sq_kappas %1% at index %2%") % sq_kapas % k_index);
	  construct_scaled_filter_coefficients_3D(new_coeffs, filter_coefficients,sq_kapas);
		  filter_lookup[k_index] = new ArrayFilter3DUsingConvolution<float>(new_coeffs);
		  all_filter_coefficients[k][j][i] = filter_lookup[k_index];
		  //new ArrayFilter3DUsingConvolution<float>(new_coeffs);	
		  
	  }
	  else 
	  {
	    all_filter_coefficients[k][j][i] = filter_lookup[k_index];
		  
	  }
		  
	
	//  all_filter_coefficients[k][j][i] = 
	  //  new ModifiedInverseAverigingArrayFilter<3,float>(inverse_filter);		
	  
	}
	else
	{	
	  sq_kapas = 0;
	//  inverse_filter = 
	  //  ModifiedInverseAverigingArrayFilter<3,float>();	  
	  all_filter_coefficients[k][j][i] =
	    new  ArrayFilter3DUsingConvolution<float>();
	  
	}
	
      }   

      
 // write_basic_interfile("kappa_coefficients_2D_SENS",*kappa_coefficients);
  delete kappa_coefficients ;

      
  for (int segment_num = start_segment_num; segment_num <= end_segment_num; ++segment_num)
  {
    delete  all_segments_for_kappa0[segment_num];
  }
    
      
      
}

// densel stuff - > apply
#if 1

template <typename elemT>
void
NonseparableSpatiallyVaryingFilters3D<elemT>:: 
virtual_apply(DiscretisedDensity<3,elemT>& out_density, const DiscretisedDensity<3,elemT>& in_density) const
{

  static map<int, shared_ptr <ArrayFilter3DUsingConvolution <float> > > filter_lookup;

  
  //the first time virtual_apply is called for this object, counter is set to 0
  static int count=0;
  // every time it's called, counter is incremented
  count++;
  info(boost::format("checking the counter  %1%") % count);
  
  const VoxelsOnCartesianGrid<float>& in_density_cast_0 =
    dynamic_cast< const VoxelsOnCartesianGrid<float>& >(in_density); 
  
  // the first set is defined for 2d separable case and the second for 3d case -- 
  // depending wether it is 2d or 3d corresponding coefficints are used. 
  static VectorWithOffset < VectorWithOffset < VectorWithOffset <shared_ptr <ArrayFilter3DUsingConvolution<float> >  > > > all_filter_coefficients;
  bool recompute_filters = false;
  //if (initial_image_filename!="1")
  //{  
    if (count ==1)
    {
      recompute_filters = true;
      all_filter_coefficients.grow(in_density_cast_0.get_min_z(),in_density_cast_0.get_max_z());    
      for (int k = in_density_cast_0.get_min_z(); k<=in_density_cast_0.get_max_z();k++)
      {
	all_filter_coefficients[k].grow(in_density_cast_0.get_min_y(),in_density_cast_0.get_max_y());
	for (int j = in_density_cast_0.get_min_y(); j<=in_density_cast_0.get_max_y();j++)      
	{
	  (all_filter_coefficients[k])[j].grow(in_density_cast_0.get_min_x(),in_density_cast_0.get_max_x()); 
	}
      }
      
      if ( precomputed_coefficients_filename=="1" )
      {
	info("Initialisation precomputed coefficients to 1");
	precomputed_coefficients_image = in_density_cast_0.get_empty_discretised_density();
        precomputed_coefficients_image->fill(1);
      }
    }
    VoxelsOnCartesianGrid<float>* precomputed_coefficients_image_cast =
	  dynamic_cast< VoxelsOnCartesianGrid<float>* >(precomputed_coefficients_image); 
    if (recompute_every_num_subiterations>0 && 
        count>recompute_from_subiteration_num &&
	count %recompute_every_num_subiterations == 0)
    {
      recompute_filters = true;
      info("recompute coefficients now\n WARNING this is specific to approach 2!");
      
      VoxelsOnCartesianGrid<float> *  normalised_bck_image_cast =
	dynamic_cast< VoxelsOnCartesianGrid<float> * > (normalised_bck_image);	 			 
      VoxelsOnCartesianGrid<float> *  sensitivity_image_cast =
	dynamic_cast< VoxelsOnCartesianGrid<float> * > (sensitivity_image);	 
      
      // WARNING this is specific to approach 2!			 
      precompute_filter_coefficients_for_second_apporach(*precomputed_coefficients_image_cast,
	in_density_cast_0,
	*sensitivity_image_cast,
	*normalised_bck_image_cast);
    }

    if (recompute_filters)
    {
      info(boost::format("Min,max in coefficients: %1%, %2%") % precomputed_coefficients_image->find_min()*rescaling_coefficient % precomputed_coefficients_image->find_max()*rescaling_coefficient);
      info("In here nonseparable");
	float max = precomputed_coefficients_image->find_max()*rescaling_coefficient;
	
#if 0
	if (count==1)
	  k_interval = max*.01F;
#endif	
	info(boost::format(" New k_interval is %1%   ") % k_interval);

	for ( int k = precomputed_coefficients_image_cast->get_min_z(); k<=precomputed_coefficients_image_cast->get_max_z();k++)
	  for ( int j = precomputed_coefficients_image_cast->get_min_y(); j<=precomputed_coefficients_image_cast->get_max_y();j++)
	    for ( int i = precomputed_coefficients_image_cast->get_min_x(); i<=precomputed_coefficients_image_cast->get_max_x();i++)
	      
	    {

	      // rescaling coefficients should be < 1 for narrowing the kernel and >1 for broadening the kernel
	      const float tmp = float((*precomputed_coefficients_image)[k][j][i]* rescaling_coefficient); 
	      //const int k_index = round(((float)1/(max(.000000001F,tmp)))/k_interval);
	      const int k_index = round((tmp)/k_interval);
#if 1
	      if (filter_lookup.find(k_index)==filter_lookup.end())

	      {
		
		info(boost::format("Now doing index %1% i.e. value %2% for tmp %3%") % k_index % k_index*k_interval % tmp);
		if (tmp >0.0000001)
		{
		  Array <3,float> new_coeffs;	 		
		  construct_scaled_filter_coefficients_3D(new_coeffs,filter_coefficients,1/tmp); 
		  filter_lookup[k_index] = new ArrayFilter3DUsingConvolution<float>(new_coeffs);
#if 0
		  {
		    char filename[1000];
		    sprintf(filename, "filter%g.hv",tmp);
		    filename[7]='_';
		    write_basic_interfile(filename, new_coeffs);
		  }
#endif
		}
		else
		{
		  filter_lookup[k_index] = new ArrayFilter3DUsingConvolution<float>();
		}
		
	      }
	      all_filter_coefficients[k][j][i] = filter_lookup[k_index];
#else	      	      
		
		info(boost::format("Now doing tmp %1%") % tmp);
		if (tmp >0.0000001)
		{
		  Array <3,float> new_coeffs;	 		
		  construct_scaled_filter_coefficients_3D(new_coeffs,filter_coefficients,1/tmp); 
		  all_filter_coefficients[k][j][i]  = new ArrayFilter3DUsingConvolution<float>(new_coeffs);
		}
		else
		{
		  all_filter_coefficients[k][j][i]  = new ArrayFilter3DUsingConvolution<float>();
		}
#endif
      }    
      
    } // end recompute_filters

    
  info("now filtering");
  
  for (int k=in_density_cast_0.get_min_z();k<=in_density_cast_0.get_max_z();k++)   
    for (int j =in_density_cast_0.get_min_y();j<=in_density_cast_0.get_max_y();j++)
      for (int i =in_density_cast_0.get_min_x();i<=in_density_cast_0.get_max_x();i++)	
      {
	Array<3,elemT> single_pixel(IndexRange3D(k,k,j,j,i,i));
	
	//cerr << "in here" << endl;
	(*all_filter_coefficients[k][j][i])(single_pixel,in_density);
	out_density[k][j][i] = single_pixel[k][j][i];
      }
      
}
#endif
     
     

template <typename elemT>
void
NonseparableSpatiallyVaryingFilters3D<elemT>:: 
virtual_apply(DiscretisedDensity<3,elemT>& density) const
{
  DiscretisedDensity<3,elemT>* tmp_density =
    density.clone();
  virtual_apply(density, *tmp_density);
  delete tmp_density;
}


template <typename elemT>
void
NonseparableSpatiallyVaryingFilters3D<elemT>::set_defaults()
{
       filter_coefficients.fill(0);
       proj_data_filename ="1";
       proj_data_ptr = NULL;
       attenuation_proj_data_filename ="1";
       initial_image_filename ="1";
       initial_image =NULL;
       sensitivity_image_filename ='1';
       sensitivity_image = NULL;
       precomputed_coefficients_filename ='1';
       normalised_bck_filename ='1';
       normalised_bck_image =NULL;
       precomputed_coefficients_image =NULL;
       attenuation_proj_data_ptr = NULL;
       mask_size = 0;
       //num_dim = 1;
       k_interval =0;
       rescaling_coefficient =0;

       recompute_from_subiteration_num = 1;
       recompute_every_num_subiterations = 0;
       
     }
     
     template <typename elemT>
       void
       NonseparableSpatiallyVaryingFilters3D<elemT>:: initialise_keymap()
     {
       parser.add_start_key("Nonseparable Spatially Varying Filters 3D");
       parser.add_key("filter_coefficients", &filter_coefficients_for_parsing);
       parser.add_key("proj_data_filename", &proj_data_filename);
       parser.add_key("attenuation_proj_data_filename", &attenuation_proj_data_filename);
       parser.add_key("initial_image_filename", &initial_image_filename);
       parser.add_key("sensitivity_image_filename", &sensitivity_image_filename);
       parser.add_key("mask_size", &mask_size);
       parser.add_key("k_interval", &k_interval);
       parser.add_key("rescaling coefficient", & rescaling_coefficient);
       parser.add_key ("precomputed_coefficients_filename", &precomputed_coefficients_filename);
       parser.add_key ("normalised_bck_filename", &normalised_bck_filename); 
       parser.add_key ("recompute_from_subiteration_num", & recompute_from_subiteration_num);
       parser.add_key ("recompute_every_num_subiterations", & recompute_every_num_subiterations);
       parser.add_stop_key("END Nonseparable Spatially Varying Filters 3D");
     }
     
template <typename elemT>
bool 
NonseparableSpatiallyVaryingFilters3D<elemT>::
post_processing()
{
  const unsigned int size_x = filter_coefficients_for_parsing[0][0].get_length();  
  const unsigned int size_y = filter_coefficients_for_parsing[0].get_length();  
  const unsigned int size_z = filter_coefficients_for_parsing.get_length();  

  const int min_index_z = -(size_z/2);  
  const int min_index_y = -(size_y/2);
  const int min_index_x = -(size_x/2);

  filter_coefficients.grow(IndexRange3D(min_index_z, min_index_z + size_z - 1,
				   min_index_y, min_index_y + size_y - 1,
  				   min_index_x, min_index_x + size_x - 1 ));

 for (int k = min_index_z; k<=filter_coefficients.get_max_index(); ++k)
  for (int j = min_index_y; j<=filter_coefficients[k].get_max_index(); ++j)
    for (int i = min_index_x; i<= filter_coefficients[k][j].get_max_index(); ++i)
    {
    filter_coefficients[k][j][i] = 
      static_cast<float>(filter_coefficients_for_parsing[k-min_index_z][j-min_index_y][i-min_index_x]);
    }
return false;
}
     
     
const char * const 
NonseparableSpatiallyVaryingFilters3D<float>::registered_name =
"Nonseparable Spatially Varying Filters 3D";
   
     
#  ifdef _MSC_VER
     // prevent warning message on reinstantiation, 
     // note that we get a linking error if we don't have the explicit instantiation below
#  pragma warning(disable:4660)
#  endif
     
     // Register this class in the ImageProcessor registry
     // static SeparableCartesianMetzImageFilter<float>::RegisterIt dummy;
     // have the above variable in a separate file, which you need t
     
template NonseparableSpatiallyVaryingFilters3D<float>;
     
     
     
END_NAMESPACE_STIR
       
#endif

