/*
    Copyright (C) 2000 PARAPET partners
    Copyright (C) 2000 - 2010-10-15, Hammersmith Imanet Ltd
    Copyright (C) 2011-07-01 -2013, Kris Thielemans
    Copyright (C) 2015, 2020, 2022 University College London
    Copyright (C) 2021-2022, Commonwealth Scientific and Industrial Research Organisation
    Copyright (C) 2021, Rutherford Appleton Laboratory STFC
    This file is part of STIR.

    SPDX-License-Identifier: Apache-2.0 AND License-ref-PARAPET-license

    See STIR/LICENSE.txt for details
*/
/*!
  \file
  \ingroup projdata  

  \brief Implementations for non-inline functions of class stir::ProjData

  \author Kris Thielemans
  \author Ashley Gillman
  \author Evgueni Ovtchinnikov
  \author Gemma Fardell
  \author PARAPET project
*/
#include "stir/ProjData.h"
#include "stir/ExamInfo.h"
#include "stir/Succeeded.h"
#include "stir/RelatedViewgrams.h"
#include "stir/Viewgram.h"
#include "stir/DataSymmetriesForViewSegmentNumbers.h"

#include "stir/utilities.h" // TODO remove (temporary for test GEAdvance)
// for read_from_file
#include "stir/IO/FileSignature.h"
#include "stir/IO/interfile.h"
#include "stir/ProjDataInterfile.h"
#include "stir/ProjDataFromStream.h" // needed for converting ProjDataFromStream* to ProjData*
#include "stir/ProjDataInMemory.h" // needed for subsets
#include "stir/ProjDataInfoSubsetByView.h"
#include "stir/Viewgram.h"


#ifndef STIR_USE_GE_IO
#include "stir/ProjDataGEAdvance.h"
#else
#include "stir_experimental/IO/GE/ProjDataVOLPET.h"
#ifdef HAVE_RDF
#include "stir_experimental/IO/GE/stir_RDF.h"
#include "stir_experimental/IO/GE/ProjDataRDF.h"
#endif
#endif // STIR_USE_GE_IO
#ifdef HAVE_IE
#include "stir_experimental/IO/GE/ProjDataIE.h"
#endif
#ifdef HAVE_HDF5
#include "stir/ProjDataGEHDF5.h"
#include "stir/IO/GEHDF5Wrapper.h"
#endif
#include "stir/IO/stir_ecat7.h"
#include "stir/ViewSegmentNumbers.h"
#include "stir/is_null_ptr.h"
#include <cstring>
#include <fstream>
#include <algorithm>

#ifndef STIR_NO_NAMESPACES
using std::istream;
using std::fstream;
using std::ios;
using std::string;
using std::vector;
#endif

START_NAMESPACE_STIR

/*! 
   This function will attempt to determine the type of projection data in the file,
   construct an object of the appropriate type, and return a pointer to 
   the object.

   The return value is a shared_ptr, to make sure that the caller will
   delete the object.

   If more than 1 projection data set is in the file, only the first is read.

   When the file is not readable for some reason, the program is aborted
   by calling error().

   Currently supported:
   <ul>
   <li> GE VOLPET data (via class ProjDataVOLPET)
   <li> Interfile (using  read_interfile_PDFS())
   <li> ECAT 7 3D sinograms and attenuation files 
   </ul>

   Developer's note: ideally the return value would be an stir::unique_ptr.
*/

shared_ptr<ProjData> 
ProjData::
read_from_file(const string& filename,
	       const std::ios::openmode openmode)
{
  std::string actual_filename = filename;
  // parse filename to see if it's like filename,options
  {
    const std::size_t comma_pos = filename.find(',');
    if (comma_pos != std::string::npos)
      {
	actual_filename.resize(comma_pos);
      }
  }

  fstream * input = new fstream(actual_filename.c_str(), openmode | ios::binary);
  if (! *input)
    error("ProjData::read_from_file: error opening file %s", actual_filename.c_str());

  const FileSignature file_signature(actual_filename);
  const char * signature = file_signature.get_signature();

  // GE Advance
  if (strncmp(signature, "2D3D", 4) == 0)
  {
//    if (ask("Read with old code (Y) or new (N)?",false))
#ifndef STIR_USE_GE_IO 
      {
#ifndef NDEBUG
	warning("ProjData::read_from_file trying to read %s as GE Advance file", 
		filename.c_str());
#endif
	return shared_ptr<ProjData>( new ProjDataGEAdvance(input) );
      }
      //else
#else // use VOLPET
      {
#ifndef NDEBUG
	warning("ProjData::read_from_file trying to read %s as GE VOLPET file", 
		filename.c_str());
#endif
	delete input;// TODO no longer use pointer after getting rid of ProjDataGEAdvance
	return shared_ptr<ProjData>( new GE_IO::ProjDataVOLPET(filename, openmode) );
      }
#endif // STIR_USE_GE_IO to differentiate between Advance and VOLPET code
  }

  delete input;// TODO no longer use pointer after getting rid of ProjDataGEAdvance

#ifdef HAVE_IE
  // GE IE file format 
  if (GE_IO::is_IE_signature(signature))
    {
#ifndef NDEBUG
      warning("ProjData::read_from_file trying to read %s as GE IE file", 
	      filename.c_str());
#endif
      return shared_ptr<ProjData>( new GE_IO::ProjDataIE(filename) );
    }
#endif // HAVE_IE
      

#ifdef HAVE_LLN_MATRIX
  // ECAT 7
  if (strncmp(signature, "MATRIX", 6) == 0)
  {
#ifndef NDEBUG
    warning("ProjData::read_from_file trying to read %s as ECAT7", filename.c_str());
#endif
    USING_NAMESPACE_ECAT;
    USING_NAMESPACE_ECAT7;

    if (is_ECAT7_emission_file(actual_filename) || is_ECAT7_attenuation_file(actual_filename))
    {
      warning("\nReading frame 1, gate 1, data 0, bed 0 from file %s",
	      actual_filename.c_str());
      shared_ptr<ProjData> proj_data_sptr(ECAT7_to_PDFS(filename, /*frame_num, gate_num, data_num, bed_num*/1,1,0,0));
      return proj_data_sptr;
    }
    else
    {
      if (is_ECAT7_file(actual_filename))
	warning("ProjData::read_from_file ECAT7 file %s is of unsupported file type", actual_filename.c_str());
    }
  }
#endif // HAVE_LLN_MATRIX

  // Interfile
  if (is_interfile_signature(signature))
  {
#ifndef NDEBUG
    warning("ProjData::read_from_file trying to read %s as Interfile", filename.c_str());
#endif
    shared_ptr<ProjData> ptr(read_interfile_PDFS(filename, openmode));
    if (!is_null_ptr(ptr))
      return ptr;
  }


#if defined(STIR_USE_GE_IO) && defined(HAVE_RDF)
  if (GE_IO::is_RDF_file(actual_filename))
    {
#ifndef NDEBUG
      warning("ProjData::read_from_file trying to read %s as RDF", filename.c_str());
#endif
      shared_ptr<ProjData> ptr(new GE_IO::ProjDataRDF(filename));
      if (!is_null_ptr(ptr))
	return ptr;
  }
#endif // RDF
      
#ifdef HAVE_HDF5
  if (GE::RDF_HDF5::GEHDF5Wrapper::check_GE_signature(actual_filename))
    {
#ifndef NDEBUG
      warning("ProjData::read_from_file trying to read %s as GE HDF5", filename.c_str());
#endif
      shared_ptr<ProjData> ptr(new GE::RDF_HDF5::ProjDataGEHDF5(filename));
      if (!is_null_ptr(ptr))
	return ptr;
  }
#endif // GE HDF5

  error("\nProjData::read_from_file could not read projection data %s.\n"
	"Unsupported file format? Aborting.",
	  filename.c_str());
  // need to return something to satisfy the compiler, but we never get here
  shared_ptr<ProjData> null_ptr;
  return null_ptr;
}

//void
//ProjData::set_exam_info(ExamInfo const& new_exam_info)
//{
//  this->exam_info_sptr.reset(new ExamInfo(new_exam_info));
//}


unique_ptr<ProjDataInMemory>
ProjData::get_subset(const std::vector<int>& views) const
{
  auto subset_proj_data_info_sptr =
    std::make_shared<ProjDataInfoSubsetByView>(proj_data_info_sptr, views);
  unique_ptr<ProjDataInMemory> subset_proj_data_uptr(new ProjDataInMemory(exam_info_sptr, subset_proj_data_info_sptr));

  //TODOTOF loop here
  for (int segment_num=get_min_segment_num(); segment_num<=get_max_segment_num(); ++segment_num)
    {
      for (int subset_view_num=0; subset_view_num < static_cast<int>(views.size()); ++subset_view_num)
        {
          const Viewgram<float> viewgram = this->get_viewgram(views[subset_view_num], segment_num);
          // construct new one with data from viewgram, but appropriate meta-data
          const Viewgram<float> subset_viewgram(viewgram, subset_proj_data_info_sptr, subset_view_num, segment_num);
          if (subset_proj_data_uptr->set_viewgram(subset_viewgram) != Succeeded::yes)
            error("ProjData::get_subset failed to set a viewgram");
        }
    }

  return subset_proj_data_uptr;
}

  
Viewgram<float> 
ProjData::get_empty_viewgram(const int view_num, const int segment_num, 
			     const bool make_num_tangential_poss_odd) const
{
  return
    proj_data_info_sptr->get_empty_viewgram(view_num, segment_num, make_num_tangential_poss_odd);
}

Sinogram<float>
ProjData::get_empty_sinogram(const int ax_pos_num, const int segment_num,
			     const bool make_num_tangential_poss_odd) const
{
  return
    proj_data_info_sptr->get_empty_sinogram(ax_pos_num, segment_num, make_num_tangential_poss_odd);
}


SegmentBySinogram<float>
ProjData::get_empty_segment_by_sinogram(const int segment_num, 
      const bool make_num_tangential_poss_odd) const
{
  return
    proj_data_info_sptr->get_empty_segment_by_sinogram(segment_num, make_num_tangential_poss_odd);
}  


SegmentByView<float>
ProjData::get_empty_segment_by_view(const int segment_num, 
				   const bool make_num_tangential_poss_odd) const
{
  return
    proj_data_info_sptr->get_empty_segment_by_view(segment_num, make_num_tangential_poss_odd);

}

RelatedViewgrams<float> 
ProjData::get_empty_related_viewgrams(const ViewSegmentNumbers& view_segmnet_num,
                   //const int view_num, const int segment_num,
		   const shared_ptr<DataSymmetriesForViewSegmentNumbers>& symmetries_used,
		   const bool make_num_tangential_poss_odd) const
{
  return
    proj_data_info_sptr->get_empty_related_viewgrams(view_segmnet_num, symmetries_used, make_num_tangential_poss_odd);
}


RelatedViewgrams<float> 
ProjData::get_related_viewgrams(const ViewSegmentNumbers& view_segmnet_num,
                   //const int view_num, const int segment_num,
		   const shared_ptr<DataSymmetriesForViewSegmentNumbers>& symmetries_used,
		   const bool make_num_bins_odd) const
{
  vector<ViewSegmentNumbers> pairs;
  symmetries_used->get_related_view_segment_numbers(
    pairs, 
    ViewSegmentNumbers(view_segmnet_num.view_num(),view_segmnet_num.segment_num())
    );

  vector<Viewgram<float> > viewgrams;
  viewgrams.reserve(pairs.size());

  for (unsigned int i=0; i<pairs.size(); i++)
  {
    // TODO optimise to get shared proj_data_info_ptr
    viewgrams.push_back(get_viewgram(pairs[i].view_num(),
                                          pairs[i].segment_num(), make_num_bins_odd));
  }

  return RelatedViewgrams<float>(viewgrams, symmetries_used);
}


Succeeded 
ProjData::set_related_viewgrams( const RelatedViewgrams<float>& viewgrams) 
{

  RelatedViewgrams<float>::const_iterator r_viewgrams_iter = viewgrams.begin();
  while( r_viewgrams_iter!=viewgrams.end())
  {
    if (set_viewgram(*r_viewgrams_iter)== Succeeded::no)
      return Succeeded::no;
    ++r_viewgrams_iter;
  }
  return Succeeded::yes;
}

#if 0
  for (int i=0; i<viewgrams.get_num_viewgrams(); ++i)
  {
    if (set_viewgram(viewgrams.get_viewgram_reference(i)) == Succeeded::no)
      return Succeeded::no;
  }
  return Succeeded::yes;
}
#endif

SegmentBySinogram<float> ProjData::get_segment_by_sinogram(const int segment_num) const
{
  SegmentBySinogram<float> segment =
    proj_data_info_sptr->get_empty_segment_by_sinogram(segment_num,false);
  // TODO optimise to get shared proj_data_info_ptr
  for (int view_num = get_min_view_num(); view_num <= get_max_view_num(); ++view_num)
    segment.set_viewgram(get_viewgram(view_num, segment_num, false));

  return segment;
}

SegmentByView<float> ProjData::get_segment_by_view(const int segment_num) const
{
  SegmentByView<float> segment =
    proj_data_info_sptr->get_empty_segment_by_view(segment_num,false);
  // TODO optimise to get shared proj_data_info_ptr
  for (int view_num = get_min_view_num(); view_num <= get_max_view_num(); ++view_num)
    segment.set_viewgram(get_viewgram(view_num, segment_num, false));

  return segment;
}

Succeeded 
ProjData::set_segment(const SegmentBySinogram<float>& segment)
{
  for (int view_num = get_min_view_num(); view_num <= get_max_view_num(); ++view_num)
  {
    if(set_viewgram(segment.get_viewgram(view_num))
        == Succeeded::no)
	return Succeeded::no;
  }
  return Succeeded::yes;
}

Succeeded 
ProjData::set_segment(const SegmentByView<float>& segment)
{
  for (int view_num = get_min_view_num(); view_num <= get_max_view_num(); ++view_num)
  {
    if(set_viewgram(segment.get_viewgram(view_num))
        == Succeeded::no)
	return Succeeded::no;
  }
  return Succeeded::yes;
}


void 
ProjData::fill(const float value)
{
  for (int segment_num = this->get_min_segment_num(); segment_num <= this->get_max_segment_num(); ++segment_num)
  {
    SegmentByView<float> segment(this->get_empty_segment_by_view(segment_num));
    segment.fill(value);
    if(this->set_segment(segment) == Succeeded::no)
      error("Error setting segment of projection data");
  }
}

void 
ProjData::fill(const ProjData& proj_data)
{
  shared_ptr<ProjDataInfo> source_proj_data_info_sptr = proj_data.get_proj_data_info_sptr()->create_shared_clone();
  source_proj_data_info_sptr->reduce_segment_range(std::max(this->get_min_segment_num(), proj_data.get_min_segment_num()),
                                                   std::min(this->get_max_segment_num(), proj_data.get_max_segment_num()));
  if ((*this->get_proj_data_info_sptr()) != (*source_proj_data_info_sptr))
      error("Filling projection data from incompatible  source");

  for (int segment_num = this->get_min_segment_num(); segment_num <= this->get_max_segment_num(); ++segment_num)
  {
    if(this->set_segment(proj_data.get_segment_by_view(segment_num))
       == Succeeded::no)
      error("Error setting segment of projection data");
  }
}

ProjData:: ProjData()
    :ExamData()
{}

ProjData::ProjData(const shared_ptr<const ExamInfo>& exam_info_sptr,
		   const shared_ptr<const ProjDataInfo>& proj_data_info_sptr)
  :ExamData(exam_info_sptr), proj_data_info_sptr(proj_data_info_sptr)
{}

Succeeded
ProjData::
write_to_file(const string& output_filename) const
{

  ProjDataInterfile out_projdata(get_exam_info_sptr(),
                 this->proj_data_info_sptr, output_filename, ios::out);

  Succeeded success=Succeeded::yes;
  for (int segment_num = proj_data_info_sptr->get_min_segment_num();
       segment_num <= proj_data_info_sptr->get_max_segment_num();
       ++segment_num)
  {
    Succeeded success_this_segment =
      out_projdata.set_segment(get_segment_by_view(segment_num));
    if (success==Succeeded::yes)
      success = success_this_segment;
  }
  return success;

}

void
ProjData::
axpby( const float a, const ProjData& x,
       const float b, const ProjData& y)
{
  xapyb(x,a,y,b);
}

void
ProjData::
xapyb(const ProjData& x, const float a,
      const ProjData& y, const float b)
{
    if (*get_proj_data_info_sptr() != *x.get_proj_data_info_sptr() ||
            *get_proj_data_info_sptr() != *y.get_proj_data_info_sptr())
        error("ProjData::xapyb: ProjDataInfo don't match");

    const int n_min = get_min_segment_num();
    const int n_max = get_max_segment_num();

    for (int s=n_min; s<=n_max; ++s)
    {
        SegmentBySinogram<float> seg = get_empty_segment_by_sinogram(s);
        const SegmentBySinogram<float> sx = x.get_segment_by_sinogram(s);
        const SegmentBySinogram<float> sy = y.get_segment_by_sinogram(s);
        seg.xapyb(sx, a, sy, b);
        set_segment(seg);
    }
}

void
ProjData::
xapyb(const ProjData& x, const ProjData& a,
      const ProjData& y, const ProjData& b)
{
    if (*get_proj_data_info_sptr() != *x.get_proj_data_info_sptr() ||
        *get_proj_data_info_sptr() != *y.get_proj_data_info_sptr() ||
        *get_proj_data_info_sptr() != *a.get_proj_data_info_sptr() ||
        *get_proj_data_info_sptr() != *b.get_proj_data_info_sptr())
        error("ProjData::xapyb: ProjDataInfo don't match");

    const int n_min = get_min_segment_num();
    const int n_max = get_max_segment_num();

    for (int s=n_min; s<=n_max; ++s)
    {
        SegmentBySinogram<float> seg = get_empty_segment_by_sinogram(s);
        const SegmentBySinogram<float> sx = x.get_segment_by_sinogram(s);
        const SegmentBySinogram<float> sy = y.get_segment_by_sinogram(s);
        const SegmentBySinogram<float> sa = a.get_segment_by_sinogram(s);
        const SegmentBySinogram<float> sb = b.get_segment_by_sinogram(s);

        seg.xapyb(sx, sa, sy, sb);
        set_segment(seg);
    }
}

void
ProjData::
sapyb(const float a, const ProjData& y, const float b)
{
  this->xapyb(*this,a,y,b);
}

void
ProjData::
sapyb(const ProjData& a, const ProjData& y,const ProjData& b)
{
  this->xapyb(*this,a,y,b);
}


std::vector<int>
ProjData::
standard_segment_sequence(const ProjDataInfo& pdi)
{
  std::vector<int> segment_sequence(pdi.get_num_segments());
  if (pdi.get_num_segments()==0)
    return segment_sequence;

  const int max_segment_num = pdi.get_max_segment_num();
  const int min_segment_num = pdi.get_min_segment_num();
  segment_sequence[0] = 0;
  unsigned idx = 1;
  int segment_num = 1;
  while (idx < segment_sequence.size())
  {
    if (segment_num<=max_segment_num)
      segment_sequence[idx++] = segment_num;
    if (-segment_num>=min_segment_num)
      segment_sequence[idx++] = -segment_num;
    ++segment_num;
  }
  return segment_sequence;
}

END_NAMESPACE_STIR
