import 'package:collection/collection.dart';
import 'package:equatable/equatable.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:kitchenowl/app.dart';
import 'package:kitchenowl/enums/update_enum.dart';
import 'package:kitchenowl/models/household.dart';
import 'package:kitchenowl/models/item.dart';
import 'package:kitchenowl/models/planner.dart';
import 'package:kitchenowl/models/recipe.dart';
import 'package:kitchenowl/models/shoppinglist.dart';
import 'package:kitchenowl/services/api/api_service.dart';
import 'package:kitchenowl/services/transaction_handler.dart';
import 'package:kitchenowl/services/transactions/household.dart';
import 'package:kitchenowl/services/transactions/planner.dart';
import 'package:kitchenowl/services/transactions/recipe.dart';
import 'package:kitchenowl/services/transactions/shoppinglist.dart';

class RecipeCubit extends Cubit<RecipeState> {
  final TransactionHandler _transactionHandler;

  RecipeCubit(Household? household, Recipe recipe, int? selectedYields)
      : this.forTesting(TransactionHandler.getInstance(), household, recipe,
            selectedYields);

  RecipeCubit.forTesting(TransactionHandler transactionHandler,
      Household? household, Recipe recipe, int? selectedYields)
      : _transactionHandler = transactionHandler,
        super(RecipeState(
          recipe: recipe,
          selectedYields: selectedYields,
          household: household,
        )) {
    refresh();
  }

  void itemSelected(RecipeItem item) {
    final Set<String> selectedItems = Set.from(state.selectedItems);
    if (!selectedItems.remove(item.name)) {
      selectedItems.add(item.name);
    }
    emit(state.copyWith(selectedItems: selectedItems));
  }

  void setSelectedYields(int selectedYields) {
    emit(state.copyWith(
      selectedYields: selectedYields,
    ));
  }

  void setUpdateState(UpdateEnum updateState) {
    emit(state.copyWith(updateState: updateState));
  }

  Future<void> refresh() async {
    final recipeFuture = _transactionHandler
        .runTransaction(TransactionRecipeGetRecipe(recipe: state.recipe));
    Future<List<ShoppingList>>? shoppingLists;
    Future<Household?>? household;
    if (state.household != null) {
      shoppingLists = _transactionHandler.runTransaction(
        TransactionShoppingListGet(household: state.household!),
        forceOffline: true,
      );
      household = _transactionHandler.runTransaction(
        TransactionHouseholdGet(household: state.household!),
      );
    }
    final (recipe, statusCode) = await recipeFuture;
    if (recipe == null) {
      emit(RecipeErrorState(recipe: state.recipe));
      return;
    }

    Recipe? inHouseholdRecipe;
    if (state.household != null && recipe.householdId != state.household!.id) {
      final allRecipes = await TransactionHandler.getInstance().runTransaction(
        TransactionRecipeGetRecipes(household: state.household!),
        forceOffline: true,
      );
      inHouseholdRecipe = allRecipes.firstWhereOrNull((r) {
        if (r.source.trim().isEmpty) return false;
        final match = RegExp(
                """(kitchenowl:\/\/|${App.currentServer})\/recipe\/${recipe.id}""")
            .matchAsPrefix(r.source.trim());

        return match != null;
      });
    }

    emit(RecipeState(
      recipe: recipe,
      updateState: state.updateState,
      selectedYields: state.selectedYields ?? recipe.yields,
      shoppingLists: shoppingLists != null ? await shoppingLists : const [],
      household: (await household) ?? state.household,
      inHouseholdRecipe: inHouseholdRecipe,
    ));
  }

  Future<void> addItemsToList([ShoppingList? shoppingList]) async {
    shoppingList ??= state.household?.defaultShoppingList;
    if (shoppingList != null) {
      await _transactionHandler
          .runTransaction(TransactionShoppingListAddRecipeItems(
        household: state.household!,
        shoppinglist: shoppingList,
        items: state.dynamicRecipe.items
            .where((item) => state.selectedItems.contains(item.name))
            .toList(),
      ));
      emit(state.copyWith(selectedItems: {}));
    }
  }

  Future<void> addRecipeToPlanner(
      {DateTime? cookingDate, bool updateOnAdd = false}) async {
    if (state.household != null) {
      await _transactionHandler.runTransaction(TransactionPlannerAddRecipe(
        household: state.household!,
        recipePlan: RecipePlan(
          recipe: state.recipe,
          cookingDate: cookingDate,
          yields: state.selectedYields != null &&
                  state.recipe.yields != state.selectedYields &&
                  state.selectedYields! > 0
              ? state.selectedYields
              : null,
        ),
      ));
      if (updateOnAdd) setUpdateState(UpdateEnum.updated);
    }
  }

  Future<Recipe?> addRecipeToHousehold() async {
    if (state.household != null) {
      final res = await ApiService.getInstance().addRecipe(
        state.household!,
        state.recipe.copyWith(
          source: "kitchenowl:///recipe/${state.recipe.id}",
          visibility: RecipeVisibility.private,
        ),
      );
      if (res != null) setUpdateState(UpdateEnum.updated);
      return res;
    }
    return null;
  }
}

final class RecipeState extends Equatable {
  final Set<String> selectedItems;
  final Recipe recipe;
  final Recipe dynamicRecipe;
  final int? selectedYields;
  final UpdateEnum updateState;
  final List<ShoppingList> shoppingLists;
  final Household? household;
  final Recipe? inHouseholdRecipe;

  RecipeState.custom({
    required this.recipe,
    required this.selectedItems,
    this.selectedYields = 0,
    this.updateState = UpdateEnum.unchanged,
    this.shoppingLists = const [],
    this.household,
    this.inHouseholdRecipe,
  }) : dynamicRecipe = recipe.withYields(selectedYields);

  RecipeState({
    required this.recipe,
    this.updateState = UpdateEnum.unchanged,
    int? selectedYields,
    this.shoppingLists = const [],
    this.household,
    this.inHouseholdRecipe,
  })  : selectedYields = selectedYields,
        dynamicRecipe = recipe.withYields(selectedYields),
        selectedItems =
            recipe.items.where((e) => !e.optional).map((e) => e.name).toSet();

  RecipeState copyWith({
    Recipe? recipe,
    Set<String>? selectedItems,
    int? selectedYields,
    UpdateEnum? updateState,
    List<ShoppingList>? shoppingLists,
    Household? household,
    Recipe? inHouseholdRecipe,
  }) =>
      RecipeState.custom(
        recipe: recipe ?? this.recipe,
        selectedItems: selectedItems ?? this.selectedItems,
        selectedYields: selectedYields ?? this.selectedYields,
        shoppingLists: shoppingLists ?? this.shoppingLists,
        updateState: updateState ?? this.updateState,
        household: household ?? this.household,
        inHouseholdRecipe: inHouseholdRecipe ?? this.inHouseholdRecipe,
      );

  @override
  List<Object?> get props => [
        recipe,
        selectedItems,
        selectedYields,
        dynamicRecipe,
        shoppingLists,
        updateState,
        household,
        inHouseholdRecipe,
      ];

  bool get isOwningHousehold =>
      household != null && recipe.householdId == household!.id;
}

final class RecipeErrorState extends RecipeState {
  RecipeErrorState({required super.recipe});
}
