import React from 'react';
import { mount } from 'enzyme';
import Anchor from '..';
import type { InternalAnchorClass } from '../Anchor';
import { sleep, render } from '../../../tests/utils';

const { Link } = Anchor;

function createGetContainer(id: string) {
  return () => {
    const container = document.getElementById(id);
    if (container == null) {
      throw new Error();
    }
    return container;
  };
}

function createDiv() {
  const root = document.createElement('div');
  document.body.appendChild(root);
  return root;
}

let idCounter = 0;
const getHashUrl = () => `Anchor-API-${idCounter++}`;

describe('Anchor Render', () => {
  const getBoundingClientRectMock = jest.spyOn(
    HTMLHeadingElement.prototype,
    'getBoundingClientRect',
  );
  const getClientRectsMock = jest.spyOn(HTMLHeadingElement.prototype, 'getClientRects');

  beforeAll(() => {
    getBoundingClientRectMock.mockReturnValue({
      width: 100,
      height: 100,
      top: 1000,
    } as DOMRect);
    getClientRectsMock.mockReturnValue({ length: 1 } as DOMRectList);
  });

  afterAll(() => {
    getBoundingClientRectMock.mockRestore();
    getClientRectsMock.mockRestore();
  });

  it('Anchor render perfectly', () => {
    const hash = getHashUrl();
    const wrapper = mount(
      <Anchor>
        <Link href={`#${hash}`} title={hash} />
      </Anchor>,
    );

    wrapper.find(`a[href="#${hash}"]`).simulate('click');
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;
    anchorInstance.handleScroll();
    expect(anchorInstance.state).not.toBe(null);
  });

  it('Anchor render perfectly for complete href - click', () => {
    const hash = getHashUrl();
    const wrapper = mount(
      <Anchor>
        <Link href={`http://www.example.com/#${hash}`} title={hash} />
      </Anchor>,
    );
    wrapper.find(`a[href="http://www.example.com/#${hash}"]`).simulate('click');
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;
    expect(anchorInstance.state.activeLink).toBe(`http://www.example.com/#${hash}`);
  });

  it('Anchor render perfectly for complete href - hash router', async () => {
    const root = createDiv();
    const scrollToSpy = jest.spyOn(window, 'scrollTo');
    mount(<div id="/faq?locale=en#Q1">Q1</div>, { attachTo: root });
    const wrapper = mount(
      <Anchor>
        <Link href="/#/faq?locale=en#Q1" title="Q1" />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;
    anchorInstance.handleScrollTo('/#/faq?locale=en#Q1');
    expect(anchorInstance.state.activeLink).toBe('/#/faq?locale=en#Q1');
    expect(scrollToSpy).not.toHaveBeenCalled();
    await sleep(1000);
    expect(scrollToSpy).toHaveBeenCalled();
  });

  it('Anchor render perfectly for complete href - scroll', () => {
    const hash = getHashUrl();
    const root = createDiv();
    mount(<div id={hash}>Hello</div>, { attachTo: root });
    const wrapper = mount(
      <Anchor>
        <Link href={`http://www.example.com/#${hash}`} title={hash} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    anchorInstance.handleScroll();
    expect(anchorInstance.state.activeLink).toBe(`http://www.example.com/#${hash}`);
  });

  it('Anchor render perfectly for complete href - scrollTo', async () => {
    const hash = getHashUrl();
    const scrollToSpy = jest.spyOn(window, 'scrollTo');
    const root = createDiv();
    mount(<div id={`#${hash}`}>Hello</div>, { attachTo: root });
    const wrapper = mount(
      <Anchor>
        <Link href={`##${hash}`} title={hash} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    anchorInstance.handleScrollTo(`##${hash}`);
    expect(anchorInstance.state.activeLink).toBe(`##${hash}`);
    const calls = scrollToSpy.mock.calls.length;
    await sleep(1000);
    expect(scrollToSpy.mock.calls.length).toBeGreaterThan(calls);
  });

  it('should remove listener when unmount', async () => {
    const hash = getHashUrl();
    const wrapper = mount(
      <Anchor>
        <Link href={`#${hash}`} title={hash} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    const removeListenerSpy = jest.spyOn((anchorInstance as any).scrollEvent, 'remove');
    wrapper.unmount();
    expect(removeListenerSpy).toHaveBeenCalled();
  });

  it('should unregister link when unmount children', () => {
    const hash = getHashUrl();
    const { container, rerender } = render(
      <Anchor>
        <Link href={`#${hash}`} title={hash} />
      </Anchor>,
    );

    expect(container.querySelectorAll('.ant-anchor-link-title')).toHaveLength(1);
    expect(container.querySelector('.ant-anchor-link-title')).toHaveAttribute('href', `#${hash}`);

    rerender(<Anchor />);
    expect(container.querySelector('.ant-anchor-link-title')).toBeFalsy();
  });

  it('should update links when link href update', async () => {
    const hash = getHashUrl();
    function AnchorUpdate({ href }: { href: string }) {
      return (
        <Anchor>
          <Link href={href} title={hash} />
        </Anchor>
      );
    }
    const wrapper = mount(<AnchorUpdate href={`#${hash}`} />);
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    if (anchorInstance == null) {
      throw new Error('anchorInstance should not be null');
    }

    expect((anchorInstance as any).links).toEqual([`#${hash}`]);
    wrapper.setProps({ href: `#${hash}_1` });
    expect((anchorInstance as any).links).toEqual([`#${hash}_1`]);
  });

  it('Anchor onClick event', () => {
    const hash = getHashUrl();
    let event;
    let link;
    const handleClick = (
      e: React.MouseEvent<HTMLElement>,
      _link: { title: React.ReactNode; href: string },
    ) => {
      event = e;
      link = _link;
    };

    const href = `#${hash}`;
    const title = hash;

    const wrapper = mount(
      <Anchor onClick={handleClick}>
        <Link href={href} title={title} />
      </Anchor>,
    );

    wrapper.find(`a[href="${href}"]`).simulate('click');

    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    anchorInstance.handleScroll();
    expect(event).not.toBe(undefined);
    expect(link).toEqual({ href, title });
  });

  it('Different function returns the same DOM', async () => {
    const hash = getHashUrl();
    const root = createDiv();
    mount(<div id={hash}>Hello</div>, { attachTo: root });
    const getContainerA = createGetContainer(hash);
    const getContainerB = createGetContainer(hash);

    const wrapper = mount(
      <Anchor getContainer={getContainerA}>
        <Link href={`#${hash}`} title={hash} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    const removeListenerSpy = jest.spyOn((anchorInstance as any).scrollEvent, 'remove');
    await sleep(1000);
    wrapper.setProps({ getContainer: getContainerB });
    expect(removeListenerSpy).not.toHaveBeenCalled();
  });

  it('Different function returns different DOM', async () => {
    const hash1 = getHashUrl();
    const hash2 = getHashUrl();
    const root = createDiv();
    mount(
      <div>
        <div id={hash1}>Hello</div>
        <div id={hash2}>World</div>
      </div>,
      { attachTo: root },
    );
    const getContainerA = createGetContainer(hash1);
    const getContainerB = createGetContainer(hash2);
    const wrapper = mount(
      <Anchor getContainer={getContainerA}>
        <Link href={`#${hash1}`} title={hash1} />
        <Link href={`#${hash2}`} title={hash2} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    const removeListenerSpy = jest.spyOn((anchorInstance as any).scrollEvent, 'remove');
    expect(removeListenerSpy).not.toHaveBeenCalled();
    await sleep(1000);
    wrapper.setProps({ getContainer: getContainerB });
    expect(removeListenerSpy).toHaveBeenCalled();
  });

  it('Same function returns the same DOM', () => {
    const hash = getHashUrl();
    const root = createDiv();
    mount(<div id={hash}>Hello</div>, { attachTo: root });
    const getContainer = createGetContainer(hash);
    const wrapper = mount(
      <Anchor getContainer={getContainer}>
        <Link href={`#${hash}`} title={hash} />
      </Anchor>,
    );
    wrapper.find(`a[href="#${hash}"]`).simulate('click');
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    (anchorInstance as any).handleScroll();
    expect(anchorInstance.state).not.toBe(null);
  });

  it('Same function returns different DOM', async () => {
    const hash1 = getHashUrl();
    const hash2 = getHashUrl();
    const root = createDiv();
    mount(
      <div>
        <div id={hash1}>Hello</div>
        <div id={hash2}>World</div>
      </div>,
      { attachTo: root },
    );
    const holdContainer = {
      container: document.getElementById(hash1),
    };
    const getContainer = () => {
      if (holdContainer.container == null) {
        throw new Error('container should not be null');
      }
      return holdContainer.container;
    };
    const wrapper = mount(
      <Anchor getContainer={getContainer}>
        <Link href={`#${hash1}`} title={hash1} />
        <Link href={`#${hash2}`} title={hash2} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    const removeListenerSpy = jest.spyOn((anchorInstance as any).scrollEvent, 'remove');
    expect(removeListenerSpy).not.toHaveBeenCalled();
    await sleep(1000);
    holdContainer.container = document.getElementById(hash2);
    wrapper.setProps({ 'data-only-trigger-re-render': true });
    expect(removeListenerSpy).toHaveBeenCalled();
  });

  it('Anchor targetOffset prop', async () => {
    const hash = getHashUrl();
    let dateNowMock;

    function dataNowMockFn() {
      let start = 0;

      const handler = () => {
        start += 1000;
        return start;
      };

      return jest.spyOn(Date, 'now').mockImplementation(handler);
    }

    dateNowMock = dataNowMockFn();

    const scrollToSpy = jest.spyOn(window, 'scrollTo');
    const root = createDiv();
    mount(<h1 id={hash}>Hello</h1>, { attachTo: root });
    const wrapper = mount(
      <Anchor>
        <Link href={`#${hash}`} title={hash} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 1000);
    dateNowMock = dataNowMockFn();

    wrapper.setProps({ offsetTop: 100 });
    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 900);
    dateNowMock = dataNowMockFn();

    wrapper.setProps({ targetOffset: 200 });
    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 800);

    dateNowMock.mockRestore();
  });

  // https://github.com/ant-design/ant-design/issues/31941
  it('Anchor targetOffset prop when contain spaces', async () => {
    const hash = `${getHashUrl()} s p a c e s`;
    let dateNowMock;

    function dataNowMockFn() {
      let start = 0;

      const handler = () => {
        start += 1000;
        return start;
      };

      return jest.spyOn(Date, 'now').mockImplementation(handler);
    }

    dateNowMock = dataNowMockFn();

    const scrollToSpy = jest.spyOn(window, 'scrollTo');
    const root = createDiv();
    mount(<h1 id={hash}>Hello</h1>, { attachTo: root });
    const wrapper = mount(
      <Anchor>
        <Link href={`#${hash}`} title={hash} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 1000);
    dateNowMock = dataNowMockFn();

    wrapper.setProps({ offsetTop: 100 });
    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 900);
    dateNowMock = dataNowMockFn();

    wrapper.setProps({ targetOffset: 200 });
    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 800);

    dateNowMock.mockRestore();
  });

  it('Anchor onChange prop', async () => {
    const hash1 = getHashUrl();
    const hash2 = getHashUrl();
    const onChange = jest.fn();
    const wrapper = mount(
      <Anchor onChange={onChange}>
        <Link href={`#${hash1}`} title={hash1} />
        <Link href={`#${hash2}`} title={hash2} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    expect(onChange).toHaveBeenCalledTimes(1);
    anchorInstance.handleScrollTo(hash2);
    expect(onChange).toHaveBeenCalledTimes(2);
    expect(onChange).toHaveBeenCalledWith(hash2);
  });

  it('invalid hash', async () => {
    const wrapper = mount(
      <Anchor>
        <Link href="notexsited" title="title" />
      </Anchor>,
    );

    wrapper.find(`a[href="notexsited"]`).simulate('click');

    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    anchorInstance.handleScrollTo('notexsited');
    expect(anchorInstance.state).not.toBe(null);
  });

  it('test edge case when getBoundingClientRect return zero size', async () => {
    getBoundingClientRectMock.mockReturnValue({
      width: 0,
      height: 0,
      top: 1000,
    } as DOMRect);
    const hash = getHashUrl();
    let dateNowMock;

    function dataNowMockFn() {
      let start = 0;

      const handler = () => {
        start += 1000;
        return start;
      };

      return jest.spyOn(Date, 'now').mockImplementation(handler);
    }

    dateNowMock = dataNowMockFn();

    const scrollToSpy = jest.spyOn(window, 'scrollTo');
    const root = createDiv();
    mount(<h1 id={hash}>Hello</h1>, { attachTo: root });
    const wrapper = mount(
      <Anchor>
        <Link href={`#${hash}`} title={hash} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 1000);
    dateNowMock = dataNowMockFn();

    wrapper.setProps({ offsetTop: 100 });
    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 900);
    dateNowMock = dataNowMockFn();

    wrapper.setProps({ targetOffset: 200 });
    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 800);

    dateNowMock.mockRestore();
    getBoundingClientRectMock.mockReturnValue({
      width: 100,
      height: 100,
      top: 1000,
    } as DOMRect);
  });

  it('test edge case when container is not windows', async () => {
    const hash = getHashUrl();
    let dateNowMock;

    function dataNowMockFn() {
      let start = 0;

      const handler = () => {
        start += 1000;
        return start;
      };

      return jest.spyOn(Date, 'now').mockImplementation(handler);
    }

    dateNowMock = dataNowMockFn();

    const scrollToSpy = jest.spyOn(window, 'scrollTo');
    const root = createDiv();
    mount(<h1 id={hash}>Hello</h1>, { attachTo: root });
    const wrapper = mount(
      <Anchor getContainer={() => document.body}>
        <Link href={`#${hash}`} title={hash} />
      </Anchor>,
    );
    const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 800);
    dateNowMock = dataNowMockFn();

    wrapper.setProps({ offsetTop: 100 });
    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 800);
    dateNowMock = dataNowMockFn();

    wrapper.setProps({ targetOffset: 200 });
    anchorInstance.handleScrollTo(`#${hash}`);
    await sleep(30);
    expect(scrollToSpy).toHaveBeenLastCalledWith(0, 800);

    dateNowMock.mockRestore();
  });

  describe('getCurrentAnchor', () => {
    it('Anchor getCurrentAnchor prop', () => {
      const hash1 = getHashUrl();
      const hash2 = getHashUrl();
      const getCurrentAnchor = () => `#${hash2}`;
      const wrapper = mount(
        <Anchor getCurrentAnchor={getCurrentAnchor}>
          <Link href={`#${hash1}`} title={hash1} />
          <Link href={`#${hash2}`} title={hash2} />
        </Anchor>,
      );
      const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

      expect(anchorInstance.state.activeLink).toBe(`#${hash2}`);
    });

    // https://github.com/ant-design/ant-design/issues/30584
    it('should trigger onChange when have getCurrentAnchor', async () => {
      const hash1 = getHashUrl();
      const hash2 = getHashUrl();
      const onChange = jest.fn();
      const wrapper = mount(
        <Anchor onChange={onChange} getCurrentAnchor={() => hash1}>
          <Link href={`#${hash1}`} title={hash1} />
          <Link href={`#${hash2}`} title={hash2} />
        </Anchor>,
      );
      const anchorInstance = wrapper.find('Anchor').last().instance() as any as InternalAnchorClass;

      expect(onChange).toHaveBeenCalledTimes(1);
      anchorInstance.handleScrollTo(hash2);
      expect(onChange).toHaveBeenCalledTimes(2);
      expect(onChange).toHaveBeenCalledWith(hash2);
    });

    // https://github.com/ant-design/ant-design/issues/34784
    it('getCurrentAnchor have default link as argument', async () => {
      const hash1 = getHashUrl();
      const hash2 = getHashUrl();
      const getCurrentAnchor = jest.fn();
      const wrapper = mount(
        <Anchor getCurrentAnchor={getCurrentAnchor}>
          <Link href={`#${hash1}`} title={hash1} />
          <Link href={`#${hash2}`} title={hash2} />
        </Anchor>,
      );

      wrapper.find(`a[href="#${hash1}"]`).simulate('click');
      expect(getCurrentAnchor).toHaveBeenCalledWith(`#${hash1}`);
      wrapper.find(`a[href="#${hash2}"]`).simulate('click');
      expect(getCurrentAnchor).toHaveBeenCalledWith(`#${hash2}`);
    });
  });
});
