import os
import re
import shutil
import subprocess
from collections import OrderedDict
from contextlib import contextmanager
from os import path
from typing import Generator

import pytest
from alembic.config import Config as AlembicConfig
from alembic.script import ScriptDirectory
from db import db
from journalist_app import create_app
from sdconfig import SecureDropConfig
from sqlalchemy import text
from tests.utils.db_helper import reset_database

MIGRATION_PATH = path.join(path.dirname(__file__), "..", "alembic", "versions")

ALL_MIGRATIONS = [
    x.split(".")[0].split("_")[0] for x in os.listdir(MIGRATION_PATH) if x.endswith(".py")
]

WHITESPACE_REGEX = re.compile(r"\s+")


@pytest.fixture
def _reset_db(config: SecureDropConfig) -> None:
    # The config fixture creates all the models in the DB, but most alembic tests expect an
    #  empty DB, so we reset the DB via this fixture
    reset_database(config.DATABASE_FILE)


def list_migrations(cfg_path, head):
    cfg = AlembicConfig(cfg_path)
    script = ScriptDirectory.from_config(cfg)
    migrations = [x.revision for x in script.walk_revisions(base="base", head=head)]
    migrations.reverse()
    return migrations


def upgrade(alembic_config, migration):
    subprocess.check_call(["alembic", "upgrade", migration], cwd=alembic_config.parent)


def downgrade(alembic_config, migration):
    subprocess.check_call(["alembic", "downgrade", migration], cwd=alembic_config.parent)


def get_schema(app):
    with app.app_context():
        result = list(
            db.engine.execute(
                text(
                    """
            SELECT type, name, tbl_name, sql
            FROM sqlite_master
            ORDER BY type, name, tbl_name
            """
                )
            )
        )

    return {(x[0], x[1], x[2]): x[3] for x in result}


def assert_schemas_equal(left, right):
    assert list(left) == list(right), "Left and right do not contain same list of tables"
    for table, left_schema in list(left.items()):
        assert_ddl_equal(left_schema, right[table])


def assert_ddl_equal(left, right):
    """Check the "tokenized" DDL is equivalent because, because sometimes
    Alembic schemas append columns on the same line to the DDL comes out
    like:

    column1 TEXT NOT NULL, column2 TEXT NOT NULL

    and SQLAlchemy comes out:

    column1 TEXT NOT NULL,
    column2 TEXT NOT NULL

    Also, sometimes CHECK constraints are duplicated by alembic, like:
    CHECK (column IN (0, 1)),
    CHECK (column IN (0, 1)),
    So dedupe alembic's output as well

    """
    # ignore the autoindex cases
    if left is None and right is None:
        return

    left_schema = left
    right_schema = right

    # dedupe output by line
    left = "\n".join(list(OrderedDict.fromkeys(left.split("\n"))))
    right = "\n".join(list(OrderedDict.fromkeys(right.split("\n"))))

    left = [x for x in WHITESPACE_REGEX.split(left) if x]
    right = [x for x in WHITESPACE_REGEX.split(right) if x]

    # Strip commas and quotes
    left = [x.replace('"', "").replace(",", "") for x in left]
    left.sort()
    right = [x.replace('"', "").replace(",", "") for x in right]
    right.sort()

    assert left == right, f"Schemas don't match:\nLeft\n{left_schema}\nRight:\n{right_schema}"


def test_alembic_head_matches_db_models(journalist_app, alembic_config, config):
    """This test is to make sure that our database models in `models.py` are
    always in sync with the schema generated by `alembic upgrade head`.
    """
    models_schema = get_schema(journalist_app)

    reset_database(config.DATABASE_FILE)
    upgrade(alembic_config, "head")

    # Recreate the app to get a new SQLALCHEMY_DATABASE_URI
    app = create_app(config)
    alembic_schema = get_schema(app)

    # The initial migration creates the table 'alembic_version', but this is
    # not present in the schema created by `db.create_all()`.
    alembic_schema = {k: v for k, v in list(alembic_schema.items()) if k[2] != "alembic_version"}

    assert_schemas_equal(alembic_schema, models_schema)


@pytest.mark.parametrize("use_config_py", [True, False])
@pytest.mark.parametrize("migration", ALL_MIGRATIONS)
def test_alembic_migration_up_and_down(alembic_config, config, use_config_py, migration, _reset_db):
    with use_config(use_config_py):
        upgrade(alembic_config, migration)
        downgrade(alembic_config, "base")


@pytest.mark.parametrize("migration", ALL_MIGRATIONS)
def test_schema_unchanged_after_up_then_downgrade(alembic_config, config, migration, _reset_db):
    if migration == "811334d7105f":
        pytest.skip("811334d7105f_sequoia_pgp doesn't delete columns on downgrade")
    # Create the app here. Using a fixture will init the database.
    app = create_app(config)

    migrations = list_migrations(alembic_config, migration)

    if len(migrations) > 1:
        target = migrations[-2]
        upgrade(alembic_config, target)
    else:
        # The first migration is the degenerate case where we don't need to
        # get the database to some base state.
        pass

    original_schema = get_schema(app)

    upgrade(alembic_config, "+1")
    downgrade(alembic_config, "-1")

    reverted_schema = get_schema(app)

    # The initial migration is a degenerate case because it creates the table
    # 'alembic_version', but rolling back the migration doesn't clear it.
    if len(migrations) == 1:
        reverted_schema = {
            k: v for k, v in list(reverted_schema.items()) if k[2] != "alembic_version"
        }

    assert_schemas_equal(reverted_schema, original_schema)


@pytest.mark.parametrize("migration", ALL_MIGRATIONS)
def test_upgrade_with_data(alembic_config, config, migration, _reset_db):
    migrations = list_migrations(alembic_config, migration)
    if len(migrations) == 1:
        # Degenerate case where there is no data for the first migration
        return

    # Upgrade to one migration before the target stored in `migration`
    last_migration = migrations[-2]
    upgrade(alembic_config, last_migration)

    # Dynamic module import
    mod_name = f"tests.migrations.migration_{migration}"
    mod = __import__(mod_name, fromlist=["UpgradeTester"])

    # Load the test data
    upgrade_tester = mod.UpgradeTester(config=config)
    upgrade_tester.load_data()

    # Upgrade to the target
    upgrade(alembic_config, migration)

    # Make sure it applied "cleanly" for some definition of clean
    upgrade_tester.check_upgrade()


@pytest.mark.parametrize("migration", ALL_MIGRATIONS)
def test_downgrade_with_data(alembic_config, config, migration, _reset_db):
    # Upgrade to the target
    upgrade(alembic_config, migration)

    # Dynamic module import
    mod_name = f"tests.migrations.migration_{migration}"
    mod = __import__(mod_name, fromlist=["DowngradeTester"])

    # Load the test data
    downgrade_tester = mod.DowngradeTester(config=config)
    downgrade_tester.load_data()

    # Downgrade to previous migration
    downgrade(alembic_config, "-1")

    # Make sure it applied "cleanly" for some definition of clean
    downgrade_tester.check_downgrade()


@contextmanager
def use_config(use: bool) -> Generator:
    if not use:
        shutil.move("config.py", "config.py.moved")
    try:
        yield
    finally:
        if not use:
            shutil.move("config.py.moved", "config.py")
