from datetime import datetime
from typing import Union

import two_factor
import werkzeug
from db import db
from flask import Blueprint, current_app, flash, g, redirect, render_template, request, url_for
from flask_babel import gettext
from journalist_app.sessions import session
from journalist_app.utils import (
    set_diceware_password,
    set_name,
    set_pending_password,
    validate_hotp_secret,
    validate_user,
)
from markupsafe import Markup
from passphrases import PassphraseGenerator


def make_blueprint() -> Blueprint:
    view = Blueprint("account", __name__)

    @view.route("/account", methods=("GET",))
    def edit() -> str:
        password = PassphraseGenerator.get_default().generate_passphrase(
            preferred_language=g.localeinfo.language
        )
        # Store password in session for future verification
        set_pending_password(session.get_user(), password)
        return render_template("edit_account.html", password=password)

    @view.route("/change-name", methods=("POST",))
    def change_name() -> werkzeug.Response:
        first_name = request.form.get("first_name")
        last_name = request.form.get("last_name")
        set_name(session.get_user(), first_name, last_name)
        return redirect(url_for("account.edit"))

    @view.route("/new-password", methods=("POST",))
    def new_password() -> werkzeug.Response:
        user = session.get_user()
        current_password = request.form.get("current_password")
        token = request.form.get("token")
        error_message = gettext("Incorrect password or two-factor code.")
        # If the user is validated, change their password
        if validate_user(user.username, current_password, token, error_message):
            password = request.form.get("password")
            if set_diceware_password(user, password):
                current_app.session_interface.logout_user(user.id)  # type: ignore
                return redirect(url_for("main.login"))
        return redirect(url_for("account.edit"))

    @view.route("/verify-2fa-totp", methods=("POST",))
    def new_two_factor_totp() -> Union[str, werkzeug.Response]:
        """
        After (re)setting a user's 2FA TOTP, allow them to verify the newly generated code.

        We don't want users to be able to see their TOTP secret after generation, so it must
        be supplied in the form body, generated by another endpoint. The provided token is
        then verified against the supplied secret.
        """
        token = request.form["token"]
        # NOTE: We only use the session for getting the user's name for the QR code
        # and don't fetch any secrets from it.
        username = session.get_user().username
        otp_secret = request.form["otp_secret"]
        totp = two_factor.TOTP(otp_secret)
        try:
            # Note: this intentionally doesn't prevent replay attacks, since we just want
            # to make sure they have the right token
            totp.verify(token, datetime.utcnow())
            flash(
                gettext("Your two-factor credentials have been reset successfully."),
                "notification",
            )
            return redirect(url_for("account.edit"))

        except two_factor.OtpTokenInvalid:
            flash(
                gettext("There was a problem verifying the two-factor code. Please try again."),
                "error",
            )

        return render_template(
            "account_new_two_factor_totp.html",
            qrcode=Markup(totp.qrcode_svg(username).decode()),
            otp_secret=otp_secret,
            formatted_otp_secret=two_factor.format_secret(otp_secret),
        )

    @view.route("/reset-2fa-totp", methods=["POST"])
    def reset_two_factor_totp() -> str:
        session.get_user().is_totp = True
        session.get_user().regenerate_totp_shared_secret()
        db.session.commit()
        new_otp_secret = session.get_user().otp_secret
        return render_template(
            "account_new_two_factor_totp.html",
            qrcode=Markup(session.get_user().totp.qrcode_svg(session.get_user().username).decode()),
            otp_secret=new_otp_secret,
            formatted_otp_secret=two_factor.format_secret(new_otp_secret),
        )

    @view.route("/verify-2fa-hotp", methods=("POST",))
    def new_two_factor_hotp() -> Union[str, werkzeug.Response]:
        """
        After (re)setting a user's 2FA HOTP, allow them to verify the newly generated code.

        This works differently than the analogous TOTP endpoint, as here we do verify against
        the database secret because we need to compare with and increment the counter.
        """
        user = session.get_user()
        token = request.form["token"]

        error = False

        if not user.is_totp:
            try:
                user.verify_2fa_token(token)
                flash(
                    gettext("Your two-factor credentials have been reset successfully."),
                    "notification",
                )
                return redirect(url_for("account.edit"))

            except two_factor.OtpTokenInvalid:
                error = True
        else:
            # XXX: Consider using a different error message here, or do we not want to reveal
            # if the user is using HOTP vs TOTP
            error = True

        if error:
            flash(
                gettext("There was a problem verifying the two-factor code. Please try again."),
                "error",
            )

        return render_template("account_new_two_factor_hotp.html", user=user)

    @view.route("/reset-2fa-hotp", methods=["POST"])
    def reset_two_factor_hotp() -> Union[str, werkzeug.Response]:
        otp_secret = request.form.get("otp_secret", None)
        if otp_secret:
            if not validate_hotp_secret(session.get_user(), otp_secret):
                return render_template("account_edit_hotp_secret.html")
            session.get_user().set_hotp_secret(otp_secret)
            db.session.commit()
            return render_template("account_new_two_factor_hotp.html", user=session.get_user())
        else:
            return render_template("account_edit_hotp_secret.html")

    return view
