import os
import re
import typing
from io import BytesIO
from pathlib import Path
from typing import BinaryIO, Dict, Optional

import pretty_bad_protocol as gnupg
from redis import Redis
from sdconfig import SecureDropConfig

import redwood

if typing.TYPE_CHECKING:
    from models import Source
    from source_user import SourceUser

# To fix https://github.com/freedomofpress/securedrop/issues/78
os.environ["USERNAME"] = "www-data"


class GpgKeyNotFoundError(Exception):
    pass


class GpgEncryptError(Exception):
    pass


class GpgDecryptError(Exception):
    pass


_default_encryption_mgr: Optional["EncryptionManager"] = None


class EncryptionManager:
    """EncryptionManager provides a high-level interface for each PGP operation we do"""

    REDIS_FINGERPRINT_HASH = "sd/crypto-util/fingerprints"
    REDIS_KEY_HASH = "sd/crypto-util/keys"

    SOURCE_KEY_UID_RE = re.compile(r"(Source|Autogenerated) Key <([-A-Za-z0-9+/=_]+)>")

    def __init__(self, gpg_key_dir: Path, journalist_pub_key: Path, redis: Redis) -> None:
        self._gpg_key_dir = gpg_key_dir
        self.journalist_pub_key = journalist_pub_key
        if not self.journalist_pub_key.exists():
            raise RuntimeError(
                f"The journalist public key does not exist at {self.journalist_pub_key}"
            )
        self._redis = redis

        # Instantiate the "main" GPG binary
        self._gpg = None

        # Instantiate the GPG binary to be used for key deletion: always delete keys without
        # invoking pinentry-mode=loopback
        # see: https://lists.gnupg.org/pipermail/gnupg-users/2016-May/055965.html
        self._gpg_for_key_deletion = None

    def gpg(self, for_deletion: Optional[bool] = False) -> gnupg.GPG:
        if for_deletion:
            if self._gpg_for_key_deletion is None:
                # GPG binary to be used for key deletion: always delete keys without
                # invoking pinentry-mode=loopback
                # see: https://lists.gnupg.org/pipermail/gnupg-users/2016-May/055965.html
                self._gpg_for_key_deletion = gnupg.GPG(
                    binary="gpg2",
                    homedir=str(self._gpg_key_dir),
                    options=["--yes", "--trust-model direct"],
                )
            return self._gpg_for_key_deletion
        else:
            if self._gpg is None:
                self._gpg = gnupg.GPG(
                    binary="gpg2",
                    homedir=str(self._gpg_key_dir),
                    options=["--pinentry-mode loopback", "--trust-model direct"],
                )
            return self._gpg

    @classmethod
    def get_default(cls) -> "EncryptionManager":
        global _default_encryption_mgr
        if _default_encryption_mgr is None:
            config = SecureDropConfig.get_current()
            _default_encryption_mgr = cls(
                gpg_key_dir=config.GPG_KEY_DIR,
                journalist_pub_key=(config.SECUREDROP_DATA_ROOT / "journalist.pub"),
                redis=Redis(decode_responses=True, **config.REDIS_KWARGS),
            )
        return _default_encryption_mgr

    def delete_source_key_pair(self, source_filesystem_id: str) -> None:
        """
        Try to delete the source's key from the filesystem.  If it's not found, either:
        (a) it doesn't exist or
        (b) the source is Sequoia-based and has its key stored in Source.pgp_public_key,
            which will be deleted when the Source instance itself is deleted.
        """
        try:
            source_key_fingerprint = self.get_source_key_fingerprint(source_filesystem_id)
        except GpgKeyNotFoundError:
            # If the source is entirely Sequoia-based, there is nothing to delete
            return

        # The subkeys keyword argument deletes both secret and public keys
        self.gpg(for_deletion=True).delete_keys(source_key_fingerprint, secret=True, subkeys=True)

        self._redis.hdel(self.REDIS_KEY_HASH, source_key_fingerprint)
        self._redis.hdel(self.REDIS_FINGERPRINT_HASH, source_filesystem_id)

    def get_journalist_public_key(self) -> str:
        return self.journalist_pub_key.read_text()

    def get_source_public_key(self, source_filesystem_id: str) -> str:
        source_key_fingerprint = self.get_source_key_fingerprint(source_filesystem_id)
        return self._get_public_key(source_key_fingerprint)

    def get_source_key_fingerprint(self, source_filesystem_id: str) -> str:
        source_key_fingerprint = self._redis.hget(self.REDIS_FINGERPRINT_HASH, source_filesystem_id)
        if source_key_fingerprint:
            return source_key_fingerprint

        # If the fingerprint was not in Redis, get it directly from GPG
        source_key_details = self._get_source_key_details(source_filesystem_id)
        source_key_fingerprint = source_key_details["fingerprint"]
        self._save_key_fingerprint_to_redis(source_filesystem_id, source_key_fingerprint)
        return source_key_fingerprint

    def get_source_secret_key_from_gpg(self, fingerprint: str, passphrase: str) -> str:
        secret_key = self.gpg().export_keys(fingerprint, secret=True, passphrase=passphrase)
        if not secret_key:
            raise GpgKeyNotFoundError()
        # Verify the secret key we got can be read and decrypted by redwood
        try:
            actual_fingerprint = redwood.is_valid_secret_key(secret_key, passphrase)
        except redwood.RedwoodError:
            # Either Sequoia can't extract the secret key or the passphrase
            # is incorrect.
            raise GpgKeyNotFoundError()
        if fingerprint != actual_fingerprint:
            # Somehow we exported the wrong key?
            raise GpgKeyNotFoundError()
        return secret_key

    def encrypt_source_message(self, message_in: str, encrypted_message_path_out: Path) -> None:
        redwood.encrypt_message(
            # A submission is only encrypted for the journalist key
            recipients=[self.get_journalist_public_key()],
            plaintext=message_in,
            destination=encrypted_message_path_out,
        )

    def encrypt_source_file(self, file_in: BinaryIO, encrypted_file_path_out: Path) -> None:
        redwood.encrypt_stream(
            # A submission is only encrypted for the journalist key
            recipients=[self.get_journalist_public_key()],
            plaintext=file_in,
            destination=encrypted_file_path_out,
        )

    def encrypt_journalist_reply(
        self, for_source: "Source", reply_in: str, encrypted_reply_path_out: Path
    ) -> None:
        redwood.encrypt_message(
            # A reply is encrypted for both the journalist key and the source key
            recipients=[for_source.public_key, self.get_journalist_public_key()],
            plaintext=reply_in,
            destination=encrypted_reply_path_out,
        )

    def decrypt_journalist_reply(self, for_source_user: "SourceUser", ciphertext_in: bytes) -> str:
        """Decrypt a reply sent by a journalist."""
        # TODO: Avoid making a database query here
        for_source = for_source_user.get_db_record()
        if for_source.pgp_secret_key is not None:
            return redwood.decrypt(
                ciphertext_in,
                secret_key=for_source.pgp_secret_key,
                passphrase=for_source_user.gpg_secret,
            ).decode()
        # In practice this should be uncreachable unless the Sequoia secret key migration failed
        ciphertext_as_stream = BytesIO(ciphertext_in)
        out = self.gpg().decrypt_file(ciphertext_as_stream, passphrase=for_source_user.gpg_secret)
        if not out.ok:
            raise GpgDecryptError(out.stderr)

        return out.data.decode("utf-8")

    def _get_source_key_details(self, source_filesystem_id: str) -> Dict[str, str]:
        for key in self.gpg().list_keys():
            for uid in key["uids"]:
                if source_filesystem_id in uid and self.SOURCE_KEY_UID_RE.match(uid):
                    return key
        raise GpgKeyNotFoundError()

    def _save_key_fingerprint_to_redis(
        self, source_filesystem_id: str, source_key_fingerprint: str
    ) -> None:
        self._redis.hset(self.REDIS_FINGERPRINT_HASH, source_filesystem_id, source_key_fingerprint)

    def _get_public_key(self, key_fingerprint: str) -> str:
        # First try to fetch the public key from Redis
        public_key = self._redis.hget(self.REDIS_KEY_HASH, key_fingerprint)
        if public_key:
            return public_key

        # Then directly from GPG
        public_key = self.gpg().export_keys(key_fingerprint)
        if not public_key:
            raise GpgKeyNotFoundError()

        self._redis.hset(self.REDIS_KEY_HASH, key_fingerprint, public_key)
        return public_key
