/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kafka.server

import kafka.cluster.Partition
import kafka.coordinator.group.GroupCoordinator
import kafka.coordinator.transaction.TransactionCoordinator
import kafka.network.RequestChannel
import kafka.server.QuotaFactory.QuotaManagers
import org.apache.kafka.common.errors.ClusterAuthorizationException
import org.apache.kafka.common.internals.Topic
import org.apache.kafka.common.network.Send
import org.apache.kafka.common.requests.{AbstractRequest, AbstractResponse}
import org.apache.kafka.common.utils.Time

object RequestHandlerHelper {

  def onLeadershipChange(groupCoordinator: GroupCoordinator,
                         txnCoordinator: TransactionCoordinator,
                         updatedLeaders: Iterable[Partition],
                         updatedFollowers: Iterable[Partition]): Unit = {
    // for each new leader or follower, call coordinator to handle consumer group migration.
    // this callback is invoked under the replica state change lock to ensure proper order of
    // leadership changes
    updatedLeaders.foreach { partition =>
      if (partition.topic == Topic.GROUP_METADATA_TOPIC_NAME)
        groupCoordinator.onElection(partition.partitionId, partition.getLeaderEpoch)
      else if (partition.topic == Topic.TRANSACTION_STATE_TOPIC_NAME)
        txnCoordinator.onElection(partition.partitionId, partition.getLeaderEpoch)
    }

    updatedFollowers.foreach { partition =>
      if (partition.topic == Topic.GROUP_METADATA_TOPIC_NAME)
        groupCoordinator.onResignation(partition.partitionId, Some(partition.getLeaderEpoch))
      else if (partition.topic == Topic.TRANSACTION_STATE_TOPIC_NAME)
        txnCoordinator.onResignation(partition.partitionId, Some(partition.getLeaderEpoch))
    }
  }

}

class RequestHandlerHelper(
  requestChannel: RequestChannel,
  quotas: QuotaManagers,
  time: Time
) {

  def throttle(
    quotaManager: ClientQuotaManager,
    request: RequestChannel.Request,
    throttleTimeMs: Int
  ): Unit = {
    val callback = new ThrottleCallback {
      override def startThrottling(): Unit = requestChannel.startThrottling(request)
      override def endThrottling(): Unit = requestChannel.endThrottling(request)
    }
    quotaManager.throttle(request, callback, throttleTimeMs)
  }

  def handleError(request: RequestChannel.Request, e: Throwable): Unit = {
    val mayThrottle = e.isInstanceOf[ClusterAuthorizationException] || !request.header.apiKey.clusterAction
    if (mayThrottle)
      sendErrorResponseMaybeThrottle(request, e)
    else
      sendErrorResponseExemptThrottle(request, e)
  }

  def sendErrorOrCloseConnection(
    request: RequestChannel.Request,
    error: Throwable,
    throttleMs: Int
  ): Unit = {
    val requestBody = request.body[AbstractRequest]
    val response = requestBody.getErrorResponse(throttleMs, error)
    if (response == null)
      requestChannel.closeConnection(request, requestBody.errorCounts(error))
    else
      requestChannel.sendResponse(request, response, None)
  }

  def sendForwardedResponse(request: RequestChannel.Request,
                            response: AbstractResponse): Unit = {
    // For forwarded requests, we take the throttle time from the broker that
    // the request was forwarded to
    val throttleTimeMs = response.throttleTimeMs()
    throttle(quotas.request, request, throttleTimeMs)
    requestChannel.sendResponse(request, response, None)
  }

  // Throttle the channel if the request quota is enabled but has been violated. Regardless of throttling, send the
  // response immediately.
  def sendResponseMaybeThrottle(request: RequestChannel.Request,
                                createResponse: Int => AbstractResponse): Unit = {
    // Only throttle non-forwarded requests
    val throttleTimeMs = if (!request.isForwarded) {
      maybeRecordAndGetThrottleTimeMs(request)
    } else 0
    throttle(quotas.request, request, throttleTimeMs)
    requestChannel.sendResponse(request, createResponse(throttleTimeMs), None)
  }

  def sendErrorResponseMaybeThrottle(request: RequestChannel.Request, error: Throwable): Unit = {
    // Only throttle non-forwarded requests or cluster authorization failures
    val throttleTimeMs = if (error.isInstanceOf[ClusterAuthorizationException] || !request.isForwarded) {
      maybeRecordAndGetThrottleTimeMs(request)
    } else 0
    throttle(quotas.request, request, throttleTimeMs)
    sendErrorOrCloseConnection(request, error, throttleTimeMs)
  }

  def maybeRecordAndGetThrottleTimeMs(request: RequestChannel.Request): Int = {
    val throttleTimeMs = quotas.request.maybeRecordAndGetThrottleTimeMs(request, time.milliseconds())
    request.apiThrottleTimeMs = throttleTimeMs
    throttleTimeMs
  }

  /**
   * Throttle the channel if the controller mutations quota or the request quota have been violated.
   * Regardless of throttling, send the response immediately.
   */
  def sendResponseMaybeThrottleWithControllerQuota(controllerMutationQuota: ControllerMutationQuota,
                                                   request: RequestChannel.Request,
                                                   createResponse: Int => AbstractResponse): Unit = {
    val timeMs = time.milliseconds
    val controllerThrottleTimeMs = controllerMutationQuota.throttleTime
    val requestThrottleTimeMs = quotas.request.maybeRecordAndGetThrottleTimeMs(request, timeMs)
    val maxThrottleTimeMs = Math.max(controllerThrottleTimeMs, requestThrottleTimeMs)
    // Only throttle non-forwarded requests
    val (effectiveControllerThrottleTime, effectiveRequestThrottleTime) = if (maxThrottleTimeMs > 0 && !request.isForwarded) {
      request.apiThrottleTimeMs = maxThrottleTimeMs
      if (controllerThrottleTimeMs > requestThrottleTimeMs) {
        (controllerThrottleTimeMs, 0)
      } else {
        (0, requestThrottleTimeMs)
      }
    } else {
      (0, 0)
    }
    throttle(quotas.controllerMutation, request, effectiveControllerThrottleTime)
    throttle(quotas.request, request, effectiveRequestThrottleTime)

    requestChannel.sendResponse(request, createResponse(maxThrottleTimeMs), None)
  }

  def sendResponseExemptThrottle(request: RequestChannel.Request,
                                 response: AbstractResponse,
                                 onComplete: Option[Send => Unit] = None): Unit = {
    quotas.request.maybeRecordExempt(request)
    requestChannel.sendResponse(request, response, onComplete)
  }

  def sendErrorResponseExemptThrottle(request: RequestChannel.Request, error: Throwable): Unit = {
    quotas.request.maybeRecordExempt(request)
    sendErrorOrCloseConnection(request, error, 0)
  }

  def sendNoOpResponseExemptThrottle(request: RequestChannel.Request): Unit = {
    quotas.request.maybeRecordExempt(request)
    requestChannel.sendNoOpResponse(request)
  }

}
