import json
import re
import textwrap
from dataclasses import dataclass
from typing import TypedDict

import anyio
import yaml
from slugify import slugify

_REPO_ROOT = anyio.Path(__file__).parent.parent
_EXAMPLES_DIR = _REPO_ROOT / "examples"
_MDX_EXAMPLES_DIR = _REPO_ROOT / "docs" / "v3" / "examples"
_RE_NEWLINE = re.compile(r"\r?\n")
_RE_FRONTMATTER = re.compile(r"^---$", re.MULTILINE)
_GITHUB_BASE_URL = "https://github.com/PrefectHQ/prefect/blob/main/"
_AUTOGENERATED_NOTE = textwrap.dedent("""
{/*
This page is automatically generated via the `generate_example_pages.py` script. Any changes to this page will be overwritten.
*/}
""")


@dataclass
class Example:
    path: anyio.Path
    title: str
    description: str
    icon: str
    keywords: list[str]
    order: int | None = None


class Frontmatter(TypedDict, total=False):
    title: str
    description: str
    icon: str
    keywords: list[str]
    order: int


async def get_examples() -> list[Example]:
    examples: list[Example] = []
    async for file in _EXAMPLES_DIR.iterdir():
        if await file.is_file() and file.suffix == ".py":
            text = await file.read_text()
            example_frontmatter = extract_front_matter(text)
            if not example_frontmatter:
                continue
            examples.append(
                Example(
                    path=file,
                    title=example_frontmatter.get("title", ""),
                    description=example_frontmatter.get("description", ""),
                    icon=example_frontmatter.get("icon", ""),
                    keywords=example_frontmatter.get("keywords", []),
                    order=example_frontmatter.get("order"),
                )
            )
    # Sort examples by order field (if present), then by filename
    examples.sort(
        key=lambda ex: (
            ex.order if ex.order is not None else float("inf"),
            ex.path.name,
        )
    )
    return examples


async def convert_example_to_mdx_page(example: Example) -> str:
    """Render a Python code example to Markdown documentation format."""

    content = await example.path.read_text()

    # Extract frontmatter to check for custom github_url
    example_frontmatter = extract_front_matter(content)

    lines = _RE_NEWLINE.split(content)
    markdown: list[str] = []
    code: list[str] = []
    for line in lines:
        if line == "#" or line.startswith("# "):
            if code:
                markdown.extend(["```python", *code, "```", ""])
                code = []
            markdown.append(line[2:])
        else:
            if markdown and markdown[-1]:
                markdown.append("")
            if code or line:
                code.append(line)

    if code:
        markdown.extend(["```python", *code, "```", ""])

    text = "\n".join(markdown)
    if _RE_FRONTMATTER.match(text):
        # Strip out frontmatter from text.
        if match := _RE_FRONTMATTER.search(text, 4):
            # Use custom github_url if provided, otherwise use default
            github_url = example_frontmatter.get(
                "github_url",
                f"{_GITHUB_BASE_URL}{example.path.relative_to(_REPO_ROOT)}",
            )

            # Use custom link text if this is an external URL
            link_text = (
                "View full project on GitHub"
                if example_frontmatter.get("github_url")
                else "View on GitHub"
            )

            # Using raw HTML for precise placement; most Markdown/MDX renderers will
            # preserve the styling while allowing fallback to a plain link if HTML
            # is stripped.
            github_button = (
                f'<a href="{github_url}" target="_blank">{link_text}</a>\n\n'
            )

            frontmatter = "---\n"

            for line in text[: match.end()].split("\n"):
                if line.startswith(("title:", "description:", "icon:", "keywords:")):
                    frontmatter += line + "\n"

            frontmatter += "---\n\n"

            # Insert the button at the very top of the document.
            text = (
                frontmatter
                + _AUTOGENERATED_NOTE
                + github_button
                + text[match.end() + 1 :]
            )

    return text


def extract_front_matter(text: str) -> Frontmatter:
    # find the block between the first two "# ---" lines
    m = re.search(
        r"^(?:# ---\s*\n)(.*?)(?:\n# ---)", text, flags=re.DOTALL | re.MULTILINE
    )
    if not m:
        return {}
    # strip leading "# " from each line
    fm = "\n".join(line.lstrip("# ").rstrip() for line in m.group(1).splitlines())
    return yaml.safe_load(fm)


async def generate_index_page(examples: list[Example]) -> str:
    TOP = textwrap.dedent(f"""---
title: Overview
icon: bars
---

{_AUTOGENERATED_NOTE}
Have an example to share? Check out our [contributing guide](/contribute/docs-contribute#contributing-examples) to get started.

<CardGroup cols={{3}}>
""")
    BOTTOM = textwrap.dedent("""
    </CardGroup>
    """)

    return (
        TOP
        + "\n".join(
            f"""
        <Card title="{example.title}" icon="{example.icon}" href="/v3/examples/{slugify(example.path.stem)}">
            {example.description}
        </Card>
"""
            for example in examples
        )
        + BOTTOM
    )


async def update_docs_json(examples: list[Example]) -> None:
    docs_json_path = _REPO_ROOT / "docs" / "docs.json"
    docs_json = await docs_json_path.read_text()
    docs_json = json.loads(docs_json)
    tabs = docs_json["navigation"]["tabs"]
    for tab in tabs:
        if tab["tab"] == "Examples":
            tab["pages"] = [
                "v3/examples/index",
                {
                    "group": "Examples",
                    "pages": [
                        f"v3/examples/{slugify(example.path.stem)}"
                        for example in examples
                    ],
                },
            ]
            break
    docs_json["navigation"]["tabs"] = tabs
    await docs_json_path.write_text(json.dumps(docs_json, indent=2))


async def main() -> None:
    examples = await get_examples()
    for example in examples:
        text = await convert_example_to_mdx_page(example)
        destination_path = _MDX_EXAMPLES_DIR / f"{slugify(example.path.stem)}.mdx"
        await destination_path.write_text(text)

    index_page = await generate_index_page(examples)
    await (_MDX_EXAMPLES_DIR / "index.mdx").write_text(index_page)

    await update_docs_json(examples)


if __name__ == "__main__":
    anyio.run(main)
