package runner

import (
	_ "net/http/pprof"
	"strings"
	"time"

	"Ernuclei/internal/runner/nucleicloud"
	"Ernuclei/pkg/catalog/loader"
	"Ernuclei/pkg/core"
	"Ernuclei/pkg/output"
	"Ernuclei/pkg/protocols"
	"github.com/projectdiscovery/gologger"
	"go.uber.org/atomic"
)

// runStandardEnumeration runs standard enumeration
func (r *Runner) runStandardEnumeration(executerOpts protocols.ExecuterOptions, store *loader.Store, engine *core.Engine) (*atomic.Bool, error) {
	if r.options.AutomaticScan {
		return r.executeSmartWorkflowInput(executerOpts, store, engine)
	}
	return r.executeTemplatesInput(store, engine)
}

// runCloudEnumeration runs cloud based enumeration
func (r *Runner) runCloudEnumeration(store *loader.Store) (*atomic.Bool, error) {
	now := time.Now()
	defer func() {
		gologger.Info().Msgf("Scan execution took %s", time.Since(now))
	}()
	client := nucleicloud.New(r.options.CloudURL, r.options.CloudAPIKey)

	results := &atomic.Bool{}

	targets := make([]string, 0, r.hmapInputProvider.Count())
	r.hmapInputProvider.Scan(func(value string) {
		targets = append(targets, value)
	})
	templates := make([]string, 0, len(store.Templates()))
	for _, template := range store.Templates() {
		templates = append(templates, getTemplateRelativePath(template.Path))
	}
	taskID, err := client.AddScan(&nucleicloud.AddScanRequest{
		RawTargets:      targets,
		PublicTemplates: templates,
	})
	if err != nil {
		return results, err
	}
	gologger.Info().Msgf("Created task with ID: %s", taskID)
	time.Sleep(3 * time.Second)

	err = client.GetResults(taskID, func(re *output.ResultEvent) {
		results.CompareAndSwap(false, true)

		if outputErr := r.output.Write(re); outputErr != nil {
			gologger.Warning().Msgf("Could not write output: %s", err)
		}
		if r.issuesClient != nil {
			if err := r.issuesClient.CreateIssue(re); err != nil {
				gologger.Warning().Msgf("Could not create issue on tracker: %s", err)
			}
		}
	})
	return results, err
}

func getTemplateRelativePath(templatePath string) string {
	splitted := strings.SplitN(templatePath, "nuclei-templates", 2)
	if len(splitted) < 2 {
		return ""
	}
	return strings.TrimPrefix(splitted[1], "/")
}
